<?php namespace EmailLog\Core\Request;

use EmailLog\Addon\API\EDDUpdater;
use EmailLog\Core\Loadie;

defined( 'ABSPATH' ) || exit; // Exit if accessed directly.

/**
 * Override WordPress Plugin API.
 * This is already done by EDD_SL_Plugin_Updater for Active add-on
 * and this class does it for all in active or yet to be installed add-ons.
 *
 * @since 2.0.0
 */
class OverridePluginAPI implements Loadie {

	/**
	 * Setup actions.
	 *
	 * @inheritdoc
	 */
	public function load() {
		add_action( 'admin_init', array( $this, 'setup_updaters_for_inactive_addons' ) );

		add_filter( 'plugins_api_result', array( $this, 'add_version_to_plugin_api_response' ), 100, 3 );
	}

	/**
	 * Setup updaters for all in-active addons.
	 */
	public function setup_updaters_for_inactive_addons() {
		$email_log = email_log();
		$licenser  = $email_log->get_licenser();

		if ( is_null( $licenser ) ) {
			return;
		}

		$inactive_addons = $licenser->get_addon_list()->get_inactive_addons();

		foreach ( $inactive_addons as $inactive_addon ) {
			$license_key = $licenser->get_addon_license_key( $inactive_addon->name );

			$updater = new EDDUpdater(
				$email_log->get_store_url(),
				$inactive_addon->file,
				array(
					'version'   => $inactive_addon->get_version(),
					'license'   => $license_key,
					'item_name' => $inactive_addon->name,
					'author'    => $inactive_addon->author,
				)
			);

			$licenser->add_updater( $updater );
		}
	}

	/**
	 * Add version attribute to plugin API response.
	 *
	 * The API response generated by EDD doesn't have the version attribute and it results in some warnings.
	 * This method fixes it by manually adding the version attribute to the API response.
	 *
	 * @since 2.1.0
	 *
	 * @param object $response API Response.
	 * @param string $action   Action name.
	 * @param array  $args     Arguments for the function.
	 *
	 * @return object Modified API response.
	 */
	public function add_version_to_plugin_api_response( $response, $action, $args ) {
		if ( 'plugin_information' !== $action ) {
			return $response;
		}

		if ( ! isset( $args->slug ) || ( substr( $args->slug, 0, 10 ) != 'email-log-' ) ) {
			return $response;
		}

		if ( isset( $response->version ) ) {
			return $response;
		}

		if ( ! isset( $response->new_version ) ) {
			return $response;
		}

		$response->version = $response->new_version;

		return $response;
	}
}
