<?php

namespace Noptin\Addons_Pack\Users;

defined( 'ABSPATH' ) || exit;

/**
 * The mass mailer class for sending emails to users.
 */
class Mailer extends \Noptin_Mass_Mailer {

	/**
	 * The email sender.
	 * @var string
	 */
	protected $sender = 'wp_users';

	/**
	 * Sends a single email to a user.
	 *
	 * @param \Noptin_Newsletter_Email $campaign
	 * @param int $recipient
	 *
	 * @return bool
	 */
	public function _send( $campaign, $recipient ) {

		$user = get_user_by( 'ID', $recipient );
		$key  = '_campaign_' . $campaign->id;

		// Bail if the user is not found or is unsubscribed...
		if ( ! $user || noptin_is_email_unsubscribed( $user->user_email ) ) {
			return null;
		}

		// ... or was already sent the email.
		if ( ! $this->can_email_user( $campaign, $user ) ) {
			return null;
		}

		// Generate and send the actual email.
		noptin()->emails->newsletter->user = $user;
		$result                            = noptin()->emails->newsletter->send( $campaign, $campaign->id, $user->user_email );

		// Log the send.
		update_user_meta( $user->ID, $key, (int) $result );

		return $result;
	}

	/**
	 * Fired after a campaign is done sending.
	 *
	 * @param @param \Noptin_Newsletter_Email $campaign
	 *
	 */
	public function done_sending( $campaign ) {
		global $wpdb;

		$wpdb->delete(
			$wpdb->usermeta,
			array(
				'meta_key' => '_campaign_' . $campaign->id,
			)
		);

	}

	/**
	 * Checks if a user is valid for a given task.
	 *
	 * @param \Noptin_Newsletter_Email $campaign The current campaign.
	 * @param \WP_User $user The user to check.
	 * @return bool
	 */
	public function can_email_user( $campaign, $user ) {

		// Do not send twice.
		if ( '' !== get_user_meta( $user->ID, '_campaign_' . $campaign->id, true ) ) {
			return false;
		}

		// Prepare sender options.
		$options = $campaign->get( 'wp_users_options' );
		$options = is_array( $options ) ? $options : array();

		// Get user locale.
		if ( ! empty( $options['locale'] ) && get_user_locale( $user ) !== $options['locale'] ) {
			return false;
		}

		return apply_filters( 'noptin_users_can_email_user_for_campaign', true, $options, $user, $campaign );
	}

	/**
	 * Fetches relevant users for the campaign.
	 *
	 * @param \Noptin_Newsletter_Email $campaign
	 */
	public function _fetch_recipients( $campaign ) {

		$manual_recipients = $campaign->get_manual_recipients_ids();

		if ( ! empty( $manual_recipients ) ) {
			return $manual_recipients;
		}

		$options       = $campaign->get( 'wp_users_options' );
		$roles         = empty( $options['roles'] ) ? array() : $options['roles'];
		$exclude       = empty( $options['exclude'] ) ? array() : wp_parse_id_list( $options['exclude'] );
		$custom_fields = isset( $options['custom_fields'] ) ? $options['custom_fields'] : array();
		$meta_query    = array();

		foreach ( $custom_fields as $key => $value ) {

			$value = trim( $value );

			// Skip empty fields.
			if ( empty( $key ) || '' === $value ) {
				continue;
			}

			// Check if an operator was specified.
			// For example, if the value is ">= 10", we'll use the >= operator.
			if ( preg_match( '/^(\=|\!\=|\<\=|\>\=|\<|\>|LIKE|NOT LIKE|IN|NOT IN|BETWEEN|NOT BETWEEN|EXISTS|NOT EXISTS|REGEXP|NOT REGEXP|RLIKE)(.*)/', $value, $matches ) ) {
				$value = isset( $matches[2] ) ? trim( $matches[2] ) : '';
				$type  = is_numeric( $value ) ? 'DECIMAL' : 'CHAR';

				// Maybe convert to array.
				if ( in_array( $matches[1], array( 'IN', 'NOT IN', 'BETWEEN', 'NOT BETWEEN' ), true ) ) {
					$value = array_map( 'trim', explode( ',', $value ) );
					$type  = isset( $value[0] ) && is_numeric( $value[0] ) ? 'DECIMAL' : 'CHAR';
				}

				$meta_query[] = array(
					'key'     => $key,
					'value'   => $value,
					'compare' => trim( $matches[1] ),
					'type'    => $type,
				);
				continue;
			}

			$meta_query[] = array(
				'key'     => $key,
				'value'   => $value,
				'compare' => '=',
			);
		}

		$query = apply_filters(
			'noptin_mass_mailer_user_query',
			array(
				'fields'     => 'ID',
				'role__in'   => $roles,
				'orderby'    => 'ID',
				'exclude'    => $exclude,
				'meta_query' => $meta_query,
			),
			$campaign
		);

		return get_users( $query );

	}

	/**
	 * Filters a recipient.
	 *
	 * @param false|array $recipient
	 * @param int $recipient_id
	 *
	 * @return array
	 */
	public function filter_recipient( $recipient, $recipient_id ) {
		$user = get_userdata( $recipient_id );

		if ( empty( $user ) ) {
			return $recipient;
		}

		return array(
			'name'  => empty( $user->display_name ) ? $user->user_login : $user->display_name,
			'email' => $user->user_email,
			'url'   => add_query_arg( 'user_id', $user->ID, admin_url( 'user-edit.php' ) ),
		);
	}

	/**
	 * Displays newsletter sending options.
	 *
	 * @param Noptin_Newsletter_Email|Noptin_automated_Email $campaign
	 *
	 * @return bool
	 */
	public function display_sending_options( $campaign ) {

		// Render sender options.
		$options = $campaign->get( 'wp_users_options' );

		$fields  = array(
			'roles'   => array(
				'description' => __( 'Select the user roles to send this email to', 'noptin-addons-pack' ),
				'type'        => 'multi_checkbox',
				'options'     => array_map( 'translate_user_role', wp_list_pluck( array_reverse( get_editable_roles() ), 'name' ) ),
			),

			'exclude' => array(
				'label'       => __( 'Exclude', 'noptin-addons-pack' ),
				'type'        => 'text',
				'placeholder' => sprintf(
					// translators: %s: Example product ids.
					__( 'For example, %s', 'noptin-addons-pack' ),
					'10, 30'
				),
				'description' => __( 'Enter a comma separated list of user ids who should not receive this email', 'noptin-addons-pack' ),
			),

		);

		// Multi-lingual support.
		if ( noptin_is_multilingual() ) {
			$fields['user_locale'] = array(
				'label'   => __( 'Preferred Language', 'noptin-addons-pack' ),
				'type'    => 'select',
				'options' => array_merge(
					array(
						'' => __( 'Any', 'noptin-addons-pack' ),
					),
					noptin_get_available_languages()
				),
			);
		}

		$fields = apply_filters( 'noptin_users_sending_options', $fields, $campaign );
		$this->display_sending_fields( $campaign, 'wp_users_options', $fields );

		// Fetch custom fields.
		$custom_fields = Custom_Fields::get();
		$field_values  = isset( $options['custom_fields'] ) ? $options['custom_fields'] : array();

		foreach ( $custom_fields as $key => $field ) {

			$value = isset( $field_values[ $key ] ) ? $field_values[ $key ] : '';

			?>
				<p>
					<label>
						<strong><?php echo wp_kses_post( $field['label'] ); ?></strong>
						<?php if ( empty( $field['options'] ) ) : ?>
							<input
								type="text"
								name="noptin_email[wp_users_options][custom_fields][<?php echo esc_attr( $key ); ?>]"
								value="<?php echo esc_attr( $value ); ?>"
								placeholder="<?php esc_attr_e( 'Any', 'noptin-addons-pack' ); ?>"
								class="widefat"
							/>
						<?php else : ?>
							<select
								name="noptin_email[wp_users_options][custom_fields][<?php echo esc_attr( $key ); ?>]"
								class="widefat"
							>
								<option value="" <?php selected( '', $value ); ?>><?php esc_html_e( 'Any', 'noptin-addons-pack' ); ?></option>
								<?php foreach ( $field['options'] as $option ) : ?>
									<option value="<?php echo esc_attr( $option ); ?>" <?php selected( $option, $value ); ?>>
										<?php echo esc_html( $option ); ?>
									</option>
								<?php endforeach; ?>
							</select>
						<?php endif; ?>
					</label>
				</p>
			<?php
		}
	}

}
