<?php

namespace Noptin\Addons_Pack\Users;

/**
 * Send email to specific paid members.
 *
 * @since 1.9.0
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Send email to specific paid members.
 */
class PMPro {

	/**
	 * Class constructor.
	 */
	public function __construct() {
		add_filter( 'noptin_woocommerce_sending_options', array( $this, 'add_membership_sending_options' ) );
		add_filter( 'noptin_users_sending_options', array( $this, 'add_membership_sending_options' ) );
		add_filter( 'noptin_subscriber_sending_options', array( $this, 'add_membership_sending_options' ) );
		add_filter( 'noptin_woocommerce_can_email_customer_for_campaign', array( $this, 'can_email_customer' ), 10, 3 );
		add_filter( 'noptin_users_can_email_user_for_campaign', array( $this, 'can_email_user' ), 10, 3 );
		add_filter( 'noptin_subscribers_can_email_subscriber_for_campaign', array( $this, 'can_email_subscriber' ), 10, 3 );
	}

	/**
	 * Registers membership sending options.
	 *
	 * @param array $options membership sending options.
	 */
	public function add_membership_sending_options( $options ) {

		// Limit to customers who have memberships to specific levels.
		$options['pmpro_level'] = array(
			'label'       => __( 'Membership Level', 'noptin-addons-pack' ),
			'type'        => 'text',
			'placeholder' => __( 'Any', 'noptin-addons-pack' ),
			'description' => sprintf(
				// translators: %s: Learn more link.
				__( 'Optional. Provide a comma separated list of membership levels if you want to limit recipients by their membership levels. Leave blank to send to all members. %s', 'noptin-addons-pack' ),
				sprintf(
					'<a href="%s" target="_blank">%s</a>',
					noptin_get_upsell_url( 'guide/sending-emails/limit-recipients-by-pmpro-membership-level/', 'paid-memberships-pro', 'email-campaigns' ),
					__( 'Learn more', 'noptin-addons-pack' )
				)
			),
		);

		return $options;
	}

	/**
	 * Determines whether a customer can be emailed for a campaign.
	 *
	 * @param bool   $can_email whether a customer can be emailed.
	 * @param array  $campaign_options campaign options.
	 * @param \WC_Customer $customer customer.
	 */
	public function can_email_customer( $can_email, $campaign_options, $customer ) {

		// Abort if already can't email.
		if ( ! $can_email || empty( $campaign_options['pmpro_level'] ) ) {
			return $can_email;
		}

		if ( ! $customer->get_id() ) {
			return $can_email;
		}

		return $this->can_email_recipient( $customer->get_id(), $campaign_options );
	}

	/**
	 * Determines whether a user can be emailed for a campaign.
	 *
	 * @param bool     $can_email whether a user can be emailed.
	 * @param array    $campaign_options campaign options.
	 * @param \WP_User $user user.
	 */
	public function can_email_user( $can_email, $campaign_options, $user ) {

		// Abort if already can't email.
		if ( ! $can_email || empty( $campaign_options['pmpro_level'] ) ) {
			return $can_email;
		}

		if ( empty( $user->ID ) ) {
			return false;
		}

		return $this->can_email_recipient( $user->ID, $campaign_options );
	}

	/**
	 * Determines whether a subscriber can be emailed for a campaign.
	 *
	 * @param bool               $can_email whether a subscriber can be emailed.
	 * @param array              $campaign_options campaign options.
	 * @param \Hizzle\Noptin\DB\Subscriber $subscriber The subscriber to check.
	 */
	public function can_email_subscriber( $can_email, $campaign_options, $subscriber ) {

		// Abort if already can't email.
		if ( ! $can_email || empty( $campaign_options['pmpro_level'] ) ) {
			return $can_email;
		}

		// Abort if we don't have an email.
		if ( empty( $subscriber->get( 'email' ) ) ) {
			return false;
		}

		$user = get_user_by( 'email', $subscriber->get( 'email' ) );

		if ( empty( $user ) ) {
			return false;
		}

		return $this->can_email_recipient( $user->ID, $campaign_options );
	}

	/**
	 * Determines whether we can email the recipient.
	 *
	 * @param int   $user_id whether a user can be emailed.
	 * @param array $campaign_options campaign options.
	 * @return bool
	 */
	public function can_email_recipient( $user_id, $campaign_options ) {

		if ( empty( $campaign_options['pmpro_level'] ) ) {
			return true;
		}

		return pmpro_hasMembershipLevel( noptin_parse_list( $campaign_options['pmpro_level'], true ), $user_id );
	}
}
