<?php

namespace Noptin\Addons_Pack\Emails;

/**
 * Manages email additions.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Manages email additions.
 *
 */
class Main {

	/**
	 * @var Posts_Merge_Tag $posts_merge_tag
	 */
	public $posts_merge_tag;

	/**
	 * @var Attachments $attachments
	 */
	public $attachments;

	/**
	 * @var Migrate $migrate
	 */
	public $migrate;

	/**
	 * Stores the main component instance.
	 *
	 * @access private
	 * @var    Main $instance The main component instance.
	 * @since  1.0.0
	 */
	private static $instance = null;

	/**
	 * Get active instance
	 *
	 * @access public
	 * @since  1.0.0
	 * @return Main The main component instance.
	 */
	public static function instance() {

		if ( empty( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Class Constructor.
	 */
	public function __construct() {
		$this->posts_merge_tag = new Posts_Merge_Tag();
		$this->migrate         = new Migrate();
		$this->attachments     = new Attachments();

		$this->add_hooks();
	}

	/**
	 * Adds email hooks.
	 *
	 */
	private function add_hooks() {
		add_filter( 'noptin_automated_email_types', array( $this, 'register_periodic_email' ) );
		add_action( 'noptin_before_email_trigger_timing_metabox', array( $this, 'render_email_trigger_timing_settings' ) );
		add_filter( 'noptin_rule_action_meta_email', array( $this, 'filter_rule_description' ), 10, 2 );
		add_filter( 'noptin_automation_rule_delay', array( $this, 'delay_automation_rule' ), 10, 2 );
	}

	/**
	 * Registers the periodic email type.
	 *
	 * @param array $types An array of automated email types.
	 * @return array
	 */
	public function register_periodic_email( $types ) {

		// Insert after the post_digest type.
		return array_slice( $types, 0, 2, true ) +
			array( 'periodic' => '\Noptin\Addons_Pack\Emails\Periodic_Email' ) +
			array_slice( $types, 2, null, true );
	}

	/**
	 * Displays the timing settings.
	 *
	 * @param \Noptin_Automated_Email $campaign
	 */
	public function render_email_trigger_timing_settings( $campaign ) {

		include noptin()->plugin_path . 'includes/emails/admin/views/automations/metabox-timing.php';
	}

	/**
	 * Filters the rule description.
	 *
	 * @param array $meta
	 * @param \Noptin_Automation_Rule $rule
	 */
	public function filter_rule_description( $meta, $rule ) {

		// Prepare action settings.
		$settings = $rule->action_settings;

		// This email is not scheduled.
		if ( empty( $settings['automated_email_id'] ) ) {
			return $meta;
		}

		$campaign = new \Noptin_Automated_Email( $settings['automated_email_id'] );

		if ( ! $campaign->exists() || $campaign->sends_immediately() ) {
			return $meta;
		}

		$meta[ __( 'Delay', 'noptin-addons-pack' ) ] = (int) $campaign->get_sends_after() . ' ' . esc_html( $campaign->get_sends_after_unit( true ) );

		return $meta;

	}

	/**
	 * Sets the automation rule delay.
	 *
	 * @param int $delay
	 * @param \Noptin_Automation_Rule $rule
	 * @return int
	 */
	public function delay_automation_rule( $delay, $rule ) {

		// Prepare action settings.
		$settings = $rule->action_settings;

		// This email is not scheduled.
		if ( empty( $settings['automated_email_id'] ) ) {
			return $delay;
		}

		$campaign = new \Noptin_Automated_Email( $settings['automated_email_id'] );

		if ( ! $campaign->exists() || $campaign->sends_immediately() ) {
			return $delay;
		}

		$interval = '+' . $campaign->get_sends_after() . ' ' . $campaign->get_sends_after_unit();

		return strtotime( $interval ) - time();
	}
}
