<?php

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

use \Elementor\Core\Files\Assets\Svg\Svg_Handler;

/**
 * Walker class
 */
class Stratum_Menu_Walker extends Walker_Nav_Menu {

	protected $meta_key 		= 'stratum-menu-item';
	private $item_settings 	    = null;
	private $elementor_settings = [];
	private $mega_menu_widget   = false;
	private $curItem;

	public function __construct( $settings, $is_mega_menu_widget = false ) {
		$this->elementor_settings = $settings;
		$this->mega_menu_widget   = $is_mega_menu_widget;
	}

	public function get_svg_to_html( $svg_id = '', $wrapper = true ) {
		if ( empty( $svg_id ) ) {
			return '';
		}

		$thumbnail_id = get_post_thumbnail_id( $svg_id );
		$url 		  = wp_get_attachment_url( $thumbnail_id );

		if ( ! $url ) {
			return '';
		}

		return $this->get_image_url( $url, array( 'class' => 'stratum-menu-icon-svg' ), $wrapper );
	}

	public function get_image_url( $url = null, $attr = array(), $wrapper = true ) {
		$url = esc_url( $url );

		if ( empty( $url ) ) {
			return;
		}

		$ext  = pathinfo( $url, PATHINFO_EXTENSION );
		$attr = array_merge( [ 'alt' => '' ], $attr );

		if ( 'svg' !== $ext ) {
			return sprintf( '<img src="%1$s"%2$s>', $url, $this->get_attr( $attr ) );
		}

		$base_url = site_url( '/' );
		$svg_path = str_replace( $base_url, ABSPATH, $url );
		$key      = md5( $svg_path );
		$svg      = get_transient( $key );

		if ( ! $svg ) {
			$svg = file_get_contents( $svg_path );
		}

		if ( ! $svg ) {
			return sprintf( '<img src="%1$s"%2$s>', $url, $this->get_attr( $attr ) );
		}

		set_transient( $key, $svg, DAY_IN_SECONDS );

		if ( ! $wrapper ) {
			return $svg;
		}

		unset( $attr[ 'alt' ] );

		return sprintf( '<div%2$s>%1$s</div>', $svg, $this->get_attr( $attr ) );
	}

	public function get_attr( $attr = array() ) {

		if ( empty( $attr ) || ! is_array( $attr ) ) {
			return;
		}

		$result = '';

		foreach ( $attr as $key => $value ) {
			$result .= sprintf( ' %s="%s"', esc_attr( $key ), esc_attr( $value ) );
		}

		return $result;
	}

	/**
	 * Starts the list before the elements are added.
	 *
	 * @since 3.0.0
	 *
	 * @see Walker::start_lvl()
	 *
	 * @param string   $output Passed by reference. Used to append additional content.
	 * @param int      $depth  Depth of menu item. Used for padding.
	 * @param stdClass $args   An object of wp_nav_menu() arguments.
	 */
	public function start_lvl( &$output, $depth = 0, $args = array() ) {

		if ( 'mega' === $this->get_item_type() ) {
			return;
		}

		if ( isset( $args->item_spacing ) && 'discard' === $args->item_spacing ) {
			$t = '';
			$n = '';
		} else {
			$t = "\t";
			$n = "\n";
		}
		$indent = str_repeat( $t, $depth );

		// Default class.
		$classes = array( 'stratum-sub-menu-simple', 'stratum-sub-menu' );

		/**
		 * Filters the CSS class(es) applied to a menu list element.
		 *
		 * @since 4.8.0
		 *
		 * @param array    $classes The CSS classes that are applied to the menu `<ul>` element.
		 * @param stdClass $args    An object of `wp_nav_menu()` arguments.
		 * @param int      $depth   Depth of menu item. Used for padding.
		 */
		$class_names = join( ' ', apply_filters( 'nav_menu_submenu_css_class', $classes, $args, $depth ) );
		$class_names = $class_names ? ' class="' . esc_attr( $class_names ) . '"' : '';

		if ( $this->mega_menu_widget ) {
			$classes 	 = array( 'stratum-sub-menu-simple', 'stratum-sub-menu', 'stratum-mega-menu-sub-menu' );
			$class_names = join( ' ', apply_filters( 'nav_menu_submenu_css_class', $classes, $args, $depth ) );
			$class_names = $class_names ? ' class="' . esc_attr( $class_names ) . '"' : '';
		}

		$output .= "{$n}{$indent}<ul $class_names>{$n}";
	}

	/**
	 * Ends the list of after the elements are added.
	 *
	 * @since 3.0.0
	 *
	 * @see Walker::end_lvl()
	 *
	 * @param string   $output Passed by reference. Used to append additional content.
	 * @param int      $depth  Depth of menu item. Used for padding.
	 * @param stdClass $args   An object of wp_nav_menu() arguments.
	 */
	public function end_lvl( &$output, $depth = 0, $args = array() ) {

		if ( 'mega' === $this->get_item_type() ) {
			return;
		}

		if ( isset( $args->item_spacing ) && 'discard' === $args->item_spacing ) {
			$t = '';
			$n = '';
		} else {
			$t = "\t";
			$n = "\n";
		}
		$indent = str_repeat( $t, $depth );
		$output .= "$indent</ul>{$n}";
	}

	/**
	 * Starts the element output.
	 *
	 * @since 3.0.0
	 * @since 4.4.0 The {@see 'nav_menu_item_args'} filter was added.
	 *
	 * @see Walker::start_el()
	 *
	 * @param string   $output Passed by reference. Used to append additional content.
	 * @param WP_Post  $item   Menu item data object.
	 * @param int      $depth  Depth of menu item. Used for padding.
	 * @param stdClass $args   An object of wp_nav_menu() arguments.
	 * @param int      $id     Current item ID.
	 */
	public function start_el( &$output, $item, $depth = 0, $args = array(), $id = 0 ) {
		$this->curItem = $item;

		// Don't put any code before this!
		$this->item_settings = null;
		$this->set_item_type( $item->ID, $depth );

		if ( 'mega' === $this->get_item_type() && 0 < $depth ) {
			return;
		}

		if ( isset( $args->item_spacing ) && 'discard' === $args->item_spacing ) {
			$t = '';
			$n = '';
		} else {
			$t = "\t";
			$n = "\n";
		}

		$settings = $this->get_settings( $item->ID );
		$indent   = ( $depth ) ? str_repeat( $t, $depth ) : '';
		$classes  = empty( $item->classes ) ? array() : (array) $item->classes;

		if ( 'mega' === $this->get_item_type() && $this->mega_menu_widget ) {
			$classes[] = 'mega-menu-item';
			$classes[] = 'menu-item-has-children';
		} else {
			$classes[] = 'simple-menu-item';
		}

		// Add an active class for ancestor items
		if ( in_array( 'current-menu-ancestor', $classes ) || in_array( 'current-page-ancestor', $classes ) ) {
			$classes[] = 'current-menu-item';
		}

		$classes[] = 'menu-item-' . $item->ID;

		if ( 0 < $depth ) {
			$classes[] = 'sub-menu-item';
		}

		$classes = array_filter( $classes );
		// add "stratum-" prefix for all classes
		array_walk( $classes, array( $this, 'modify_menu_item_classes' ) );

		/**
		 * Filters the arguments for a single nav menu item.
		 *
		 * @since 4.4.0
		 *
		 * @param stdClass $args  An object of wp_nav_menu() arguments.
		 * @param WP_Post  $item  Menu item data object.
		 * @param int      $depth Depth of menu item. Used for padding.
		 */
		$args = apply_filters( 'nav_menu_item_args', $args, $item, $depth );

		/**
		 * Filters the CSS class(es) applied to a menu item's list item element.
		 *
		 * @since 3.0.0
		 * @since 4.1.0 The `$depth` parameter was added.
		 *
		 * @param array    $classes The CSS classes that are applied to the menu item's `<li>` element.
		 * @param WP_Post  $item    The current menu item.
		 * @param stdClass $args    An object of wp_nav_menu() arguments.
		 * @param int      $depth   Depth of menu item. Used for padding.
		 */

		$class_names = join( ' ', apply_filters( 'nav_menu_css_class', $classes, $item, $args, $depth ) );
		$class_names = $class_names ? ' class="' . esc_attr( $class_names ) . '"' : '';

		/**
		 * Filters the ID applied to a menu item's list item element.
		 *
		 * @since 3.0.1
		 * @since 4.1.0 The `$depth` parameter was added.
		 *
		 * @param string   $menu_id The ID that is applied to the menu item's `<li>` element.
		 * @param WP_Post  $item    The current menu item.
		 * @param stdClass $args    An object of wp_nav_menu() arguments.
		 * @param int      $depth   Depth of menu item. Used for padding.
		 */
		$id = apply_filters( 'nav_menu_item_id', 'stratum-menu-item-'. $item->ID, $item, $args, $depth );
		$id = $id ? ' id="' . esc_attr( $id ) . '"' : '';

		$output .= $indent . '<li' . $id . $class_names .'>';

		$link_classes = array('stratum-menu-item__link');

		$atts = array();
		$atts['title']  = ! empty( $item->attr_title ) ? $item->attr_title : '';
		$atts['target'] = ! empty( $item->target )     ? $item->target     : '';
		$atts['rel']    = ! empty( $item->xfn )        ? $item->xfn        : '';
		$atts['href']   = ! empty( $item->url )        ? $item->url        : '';

		$link_classes[] = ( 0 === $depth ) ? 'top-level-link'  : 'sub-level-link';

		if ( isset( $settings['menu_enable_mega_menu'] ) && $settings['menu_enable_mega_menu'] == 'yes' && $this->mega_menu_widget ) {
			$link_classes[] = 'mega-menu-enable';
		}

		$atts['class'] = implode( ' ', $link_classes );

		/**
		 * Filters the HTML attributes applied to a menu item's anchor element.
		 *
		 * @since 3.6.0
		 * @since 4.1.0 The `$depth` parameter was added.
		 *
		 * @param array $atts {
		 *     The HTML attributes applied to the menu item's `<a>` element, empty strings are ignored.
		 *
		 *     @type string $title  Title attribute.
		 *     @type string $target Target attribute.
		 *     @type string $rel    The rel attribute.
		 *     @type string $href   The href attribute.
		 * }
		 * @param WP_Post  $item  The current menu item.
		 * @param stdClass $args  An object of wp_nav_menu() arguments.
		 * @param int      $depth Depth of menu item. Used for padding.
		 */
		$atts = apply_filters( 'nav_menu_link_attributes', $atts, $item, $args, $depth );

		$attributes = '';

		foreach ( $atts as $attr => $value ) {
			if ( ! empty( $value ) ) {
				$value = ( 'href' === $attr ) ? esc_url( $value ) : esc_attr( $value );
				$attributes .= ' ' . $attr . '="' . $value . '"';
			}
		}

		/** This filter is documented in wp-includes/post-template.php */
		$title = apply_filters( 'the_title', $item->title, $item->ID );

		/**
		 * Filters a menu item's title.
		 *
		 * @since 4.4.0
		 *
		 * @param string   $title The menu item's title.
		 * @param WP_Post  $item  The current menu item.
		 * @param stdClass $args  An object of wp_nav_menu() arguments.
		 * @param int      $depth Depth of menu item. Used for padding.
		 */
		$title = apply_filters( 'nav_menu_item_title', $title, $item, $args, $depth );

		$icon = $this->generateIconHTML($settings);
		$badge = $this->generateBadgeHTML($settings);
		$arrow = $this->generateDropdownIconHTML($item, $settings, $depth);

		$link_text_style = (isset($settings['menu_item_color']) && $settings['menu_item_color'] != '' ? ' style="color:'.esc_attr($settings['menu_item_color']).';"' : '');

		$item_output = $args->before;
		$item_output .= '<div class="stratum-menu-item__wrapper">';
		$item_output .= '<div class="stratum-menu-item__link-wrapper">';
		$item_output .= $args->link_before;
		$item_output .= '<a ' . $attributes . '>';
		$item_output .= $icon;
		$item_output .= '<span class="stratum-menu-text" ' . $link_text_style . '>' . $title . '</span>';
		$item_output .= '</a>';
		$item_output .= $args->link_after;
		$item_output .= $badge;
		$item_output .= '</div>';
		$item_output .= $arrow;
		$item_output .= '</div>';
		$item_output .= $args->after;

		$is_elementor = ( isset( $_GET['elementor-preview'] ) ) ? true : false;

		$mega_item = get_post_meta( $item->ID, $this->meta_key, true );

		$mega_item = isset($mega_item['mega_menu_id']) ? $mega_item['mega_menu_id'] : '';

		if ( $this->is_mega_enabled( $item->ID ) && ! $is_elementor && $mega_item != '' && $this->mega_menu_widget ) {
			$content = '';

			if ( class_exists( 'Elementor\Plugin' ) ) {
				$elementor = Elementor\Plugin::instance();
				$content   = $elementor->frontend->get_builder_content_for_display( $mega_item );
			}

			$content = do_shortcode( $content );

			$item_output .= sprintf( '<div class="stratum-sub-mega-menu stratum-sub-menu stratum-mega-menu-sub-menu" data-template-id="%s"><div class="stratum-sub-mega-menu__wrapper">%s</div></div>', $mega_item, $content );
		}

		/**
		 * Filters a menu item's starting output.
		 *
		 * The menu item's starting output only includes `$args->before`, the opening `<a>`,
		 * the menu item's title, the closing `</a>`, and `$args->after`. Currently, there is
		 * no filter for modifying the opening and closing `<li>` for a menu item.
		 *
		 * @since 3.0.0
		 *
		 * @param string				$item_output	The menu item's starting HTML output.
		 * @param WP_Post				$item			Menu item data object.
		 * @param Stratum_Menu_Walker	$this			Current menu walker.
		 * @param int					$depth			Depth of menu item. Used for padding.
		 * @param stdClass				$args			An object of wp_nav_menu() arguments.
		 */
		$output .= apply_filters( 'stratum-menu/menu-walker/start-el', $item_output, $item, $this, $depth, $args );
	}

	/**
	 * Ends the element output, if needed.
	 *
	 * @since 3.0.0
	 *
	 * @see Walker::end_el()
	 *
	 * @param string   $output Passed by reference. Used to append additional content.
	 * @param WP_Post  $item   Page data object. Not used.
	 * @param int      $depth  Depth of page. Not Used.
	 * @param stdClass $args   An object of wp_nav_menu() arguments.
	 */
	public function end_el( &$output, $item, $depth = 0, $args = array() ) {

		if ( 'mega' === $this->get_item_type() && 0 < $depth ) {
			return;
		}

		$item_output = "</li>";
		$item_output = apply_filters( 'stratum-menu/menu-walker/end-el', $item_output, $item, $this, $depth, $args );

		$output .= $item_output;

	}

	/**
	 * Modify menu item classes list
	 *
	 * @param  string &$item
	 * @return void
	 */
	public function modify_menu_item_classes( &$item, $index ) {
		if ( 0 === $index && 'menu-item' !== $item ) {
			return;
		}

		$item = 'stratum-' . $item;
	}

	/**
	 * Store in WP Cache processed item type
	 *
	 * @param integer $item_id Current menu Item ID
	 * @param integer $depth   Current menu Item depth
	 */
	public function set_item_type( $item_id = 0, $depth = 0 ) {

		if ( 0 < $depth ) {
			return;
		}

		$item_type = 'simple';

		if ( $this->is_mega_enabled( $item_id ) ) {
			$item_type = 'mega';
		}

		wp_cache_set( 'item-type', $item_type, 'stratum-menu' );

	}

	/**
	 * Returns current item (for top level items) or parent item (for subs) type.
	 * @return [type] [description]
	 */
	public function get_item_type() {
		return wp_cache_get( 'item-type', 'stratum-menu' );
	}

	/**
	 * Check if mega menu enabled for passed item
	 *
	 * @param  int  $item_id Item ID
	 * @return boolean
	 */
	public function is_mega_enabled( $item_id = 0 ) {
		$item_settings = $this->get_settings( $item_id );

		return  $this->mega_menu_widget && (isset($item_settings['menu_enable_mega_menu']) && $item_settings['menu_enable_mega_menu'] == 'yes');
	}

	/**
	 * Get item settings
	 *
	 * @param  integer $item_id Item ID
	 * @return array
	 */
	public function get_settings( $item_id = 0 ) {

		if ( null === $this->item_settings ) {
			$this->item_settings = get_post_meta( $item_id, $this->meta_key, true );
		}

		return $this->item_settings;
	}

	/**
	 * Generates html for item icon if it is set
	 *
	 * @param array $settings Settings retrieved from menu item meta
	 * @return string Generated HTML or empty string
	 */
	private function generateIconHTML($settings) {
		$icon = '';

		// if icon set from elementor icon library
		if ( isset( $settings[ 'menu_icon_type' ] ) && $settings[ 'menu_icon_type' ] === 'icon_type' ) {
			if ( ! empty( $settings[ 'menu_item_icon' ] ) ) {
				$style = $settings[ 'menu_icon_color' ] != '' ? 'style="color: ' . esc_attr( $settings[ 'menu_icon_color' ] ) . ';"' : '';

				$icon = '<span class="stratum-menu-icon" ' . $style . '><i class="' . esc_attr( $settings[ 'menu_item_icon' ] ) .'"></i></span>';
			}
		}

		// if icon is custom svg
		if ( isset( $settings[ 'menu_icon_type' ] ) && $settings[ 'menu_icon_type' ] === 'icon_svg' ) {
			if ( ! empty( $settings[ 'menu_item_icon_svg' ] ) ) {
				$style = $settings[ 'menu_icon_color' ] != '' ? 'style="color: ' . esc_attr( $settings[ 'menu_icon_color' ] ) . ';"' : '';

				$icon = '<span class="stratum-menu-icon" ' . $style . '>' . $this->get_svg_to_html( $settings[ 'menu_item_icon_svg' ], false ) . '</span>';
			}
		}

		return $icon;
	}

	/**
	 * Generates html for item badge if it is set
	 *
	 * @param array $settings Settings retrieved from menu item meta
	 * @return string Generated HTML or empty string
	 */
	private function generateBadgeHTML($settings) {
		$badge = '';
		$color_style = '';
		$bg_style = '';

		// text color styles
		if ( isset($settings['menu_badge_color']) && $settings['menu_badge_color'] != '' ) {
			$color_style .= 'color:' . esc_attr( $settings[ 'menu_badge_color' ] ) . ';';
		}

		// bg color styles
		if ( isset($settings['menu_badge_background']) &&  $settings['menu_badge_background'] != '' ) {
			$color_style .= 'background-color:' . esc_attr( $settings[ 'menu_badge_background' ] ) . ';';
		}

		$style = sprintf('style="%1$s %2$s"', $color_style, $bg_style);

		if ( ! empty( $settings[ 'menu_badge_text' ] ) ) {

			$badge = '<span class="stratum-menu-badge" ' . $style . '>' . esc_html( $settings[ 'menu_badge_text' ] ) . '</span>';
		}

		return $badge;
	}

	/**
	 * Generates html for item dropdown icon if it is set
	 *
	 * @param WP_Post $item Menu item data object.
	 * @param array $settings Settings retrieved from menu item meta
	 * @param int $depth Depth of menu item
	 * @return string Generated HTML or empty string
	 */
	private function generateDropdownIconHTML($item, $settings, $depth = 0) {

		// prevent adding dropdown for simple menu widget
		if ( ! $this->mega_menu_widget ) {
			return '';
		}

		$arrow = '';
		$item_has_children = in_array( 'menu-item-has-children', $item->classes );
		$mega_menu_enabled = $this->mega_menu_widget && $this->is_mega_enabled( $item->ID );

		$should_add_dropdown_icon =  $item_has_children || $mega_menu_enabled;

		if ( $should_add_dropdown_icon ) {
			$arrow = '<button class="stratum-menu-dropdown"><i class="icon fa fa-angle-down"></i></button>';
		}

		return $arrow;
	}
}
