<?php

defined( 'ABSPATH' ) || exit;

/**
 * Noptin_AP_Form_Enhancements Class
 *
 * @since 1.0.0
 * @ignore
 */
class Noptin_AP_Form_Enhancements {

	protected $forms = null;

	/**
	 * Class constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {
		add_action( 'noptin_form_advanced_settings_editor', array( $this, 'display_popup_settings' ) );
		add_action( 'wp_footer', array( $this, 'display_popups' ), 5 );
		add_action( 'wp_enqueue_scripts', array( $this, 'load_scripts' ) );
    }

	/**
	 * Displays popup settings.
	 *
	 * @param Noptin_Form $form
	 */
	public function display_popup_settings( $form ) {

		$popup_types = array(

			'popup' => array(
				__( 'Popup Settings', 'noptin-addons-pack' ),
				__( 'Display this form in a popup', 'noptin-addons-pack' ),
				__( 'Show the popup', 'noptin-addons-pack' ),
			),

			'slide' => array(
				__( 'Sliding Settings', 'noptin-addons-pack' ),
				__( 'Slide this form into view', 'noptin-addons-pack' ),
				__( 'Slide this form', 'noptin-addons-pack' ),
			),

		);

		foreach ( $popup_types as $key => $labels ) {
			$settings = isset( $form->settings[ $key ] ) ? $form->settings[ $key ] : array();
			$this->display_popup_setting( $key, $labels, $settings );
		}

	}

	/**
	 * Displays settings panel.
	 *
	 * @param string $key
	 * @param array $labels
	 * @param array $settings
	 */
	public function display_popup_setting( $key, $labels, $settings ) {

		$triggers = array(
			'immeadiate'   => __( 'As soon the page loads', 'noptin-addons-pack' ),
			'before_leave' => __( 'Before the user leaves', 'noptin-addons-pack' ),
			'on_scroll'    => __( 'After the user scrolls', 'noptin-addons-pack' ),
			'after_click'  => __( 'When the user clicks on', 'noptin-addons-pack' ),
			'after_delay'  => __( 'After', 'noptin-addons-pack' ),
		);
		$trigger  = isset( $settings['trigger'] ) ? $settings['trigger'] : 'immeadiate';

		$directions = array(
			'top_left'     => __( 'Top Left', 'noptin-addons-pack' ),
			'top_right'    => __( 'Top Right', 'noptin-addons-pack' ),
			'bottom_left'  => __( 'Bottom Left', 'noptin-addons-pack' ),
			'bottom_right' => __( 'Bottom Right', 'noptin-addons-pack' ),
		);
		$direction  = isset( $settings['direction'] ) ? $settings['direction'] : 'bottom_right';

		$positions = array(
			'top'    => __( 'Top of the page', 'noptin-addons-pack' ),
			'bottom' => __( 'Bottom of the page', 'noptin-addons-pack' ),
		);
		$position  = isset( $settings['position'] ) ? $settings['position'] : 'top';
		?>

		<div class="noptin-text-wrapper form-settings-<?php echo esc_attr( $key ); ?>">
			<label for="noptin-form-enable-<?php echo esc_attr( $key ); ?>" class="noptin-field-label"><?php echo esc_html( $labels[0] ); ?></label>

			<?php noptin_hidden_field( 'noptin_form[settings][' . $key . '][enable]', 0 ); ?>
			<label>
				<input type="checkbox" id="noptin-form-enable-<?php echo esc_attr( $key ); ?>" name="noptin_form[settings][<?php echo esc_attr( $key ); ?>][enable]" value="1" <?php checked( ! empty( $settings['enable'] ) ); ?>/>
				<span class="description"><?php echo esc_html( $labels[1] ); ?></span>
			</label>

			<?php if ( 'slide' === $key ) : ?>
				<label><span style="vertical-align: middle;"><?php esc_html_e( 'from the', 'noptin-addons-pack' ); ?></span>
					<select id="noptin-form-slide-direction" name="noptin_form[settings][<?php echo esc_attr( $key ); ?>][direction]">
						<?php foreach ( $directions as $_key => $label ) : ?>
							<option value="<?php echo esc_attr( $_key ); ?>" <?php selected( $_key, $direction ); ?>><?php echo esc_html( $label ); ?></option>
						<?php endforeach; ?>
					</select>
				</label>
			<?php endif; ?>

			<?php if ( 'bar' === $key ) : ?>
				<label><span style="vertical-align: middle;"><?php esc_html_e( 'at the', 'noptin-addons-pack' ); ?></span>
					<select id="noptin-form-bar-position" name="noptin_form[settings][<?php echo esc_attr( $key ); ?>][position]">
						<?php foreach ( $positions as $_key => $label ) : ?>
							<option value="<?php echo esc_attr( $_key ); ?>" <?php selected( $_key, $position ); ?>><?php echo esc_html( $label ); ?></option>
						<?php endforeach; ?>
					</select>
				</label>
			<?php endif; ?>

			<select id="noptin-form-trigger-<?php echo esc_attr( $key ); ?>" name="noptin_form[settings][<?php echo esc_attr( $key ); ?>][trigger]">
				<?php foreach ( $triggers as $_key => $label ) : ?>
					<option value="<?php echo esc_attr( $_key ); ?>" <?php selected( $_key, $trigger ); ?>><?php echo esc_html( $label ); ?></option>
				<?php endforeach; ?>
			</select>

			<label class="noptin-<?php echo esc_attr( $key ); ?>-trigger-value on_scroll" style="<?php echo 'on_scroll' === $trigger ? '' : 'display: none;'; ?>"><input type="number" style="width: 50px;" max="100" min="0" step="0.1" name="noptin_form[settings][<?php echo esc_attr( $key ); ?>][on_scroll]" value="<?php echo isset( $settings['on_scroll'] ) ? floatval( $settings['on_scroll'] ) : '25'; ?>" /><?php esc_html_e( '% of the page', 'noptin-addons-pack' ); ?></label>
			<label class="noptin-<?php echo esc_attr( $key ); ?>-trigger-value after_click" style="<?php echo 'after_click' === $trigger ? '' : 'display: none;'; ?>"><input type="text" name="noptin_form[settings][<?php echo esc_attr( $key ); ?>][after_click]" value="<?php echo isset( $settings['after_click'] ) ? esc_attr( $settings['after_click'] ) : ''; ?>" placeholder="<?php esc_attr_e( 'Enter CSS selector', 'noptin-addons-pack' ); ?>"/></label>
			<label class="noptin-<?php echo esc_attr( $key ); ?>-trigger-value after_delay" style="<?php echo 'after_delay' === $trigger ? '' : 'display: none;'; ?>"><input type="number" style="width: 80px;" name="noptin_form[settings][<?php echo esc_attr( $key ); ?>][after_delay]" value="<?php echo isset( $settings['after_delay'] ) ? floatval( $settings['after_delay'] ) : '5'; ?>" /><?php esc_html_e( 'seconds', 'noptin-addons-pack' ); ?></label>

			<script>
				jQuery( '#noptin-form-trigger-<?php echo esc_js( $key ); ?>' ).on( 'change', function() {
					jQuery( '.noptin-<?php echo esc_js( $key ); ?>-trigger-value' ).hide();
					jQuery( '.noptin-<?php echo esc_js( $key ); ?>-trigger-value.' + jQuery( this ).val() ).show();
				});
			</script>
		</div>

		<?php
	}

	/**
	 * Displays popups on the front end
	 *
	 * @access      public
	 * @since       1.0.5
	 * @return      void
	 */
	public function display_popups() {

		$popup_types  = array( 'popup', 'bar', 'slide' );
		foreach ( $this->get_forms() as $form ) {

			// Can it be displayed?
			if ( ! $form->can_show() ) {
				continue;
			}

			// Loop through each popup type.
			foreach ( $popup_types as $type ) {

				// Is it enabled.
				if ( empty( $form->settings[ $type ]['enable'] ) ) {
					continue;
				}

				$settings  = $form->settings[ $type ];
				$trigger   = isset( $settings['trigger'] ) ? $settings['trigger'] : 'immeadiate';
				$value     = isset( $settings[ $trigger ] ) ? $settings[ $trigger ] : 'none';
				$direction = isset( $settings['direction'] ) ? $settings['direction'] : 'bottom_right';
				$position  = isset( $settings['position'] ) ? $settings['position'] : 'top';
				$atts      = array(
					'class'          => 'noptin-popup-wrapper',
					'data-key'       => $type . '-' . $form->id,
					'data-value'     => $value,
					'data-type'      => $type,
					'data-position'  => $position,
					'data-direction' => $direction,
					'data-trigger'   => $trigger,
					'tabindex'       => '-1',
				);

				?>
					<div <?php noptin_attr( 'popup_wrapper', $atts ); ?>>
						<div class="noptin-popup-inner">
							<?php $form->display(); ?>
							<span class="noptin-close-popup"></span>
						</div>
					</div>

				<?php
			}
		}

		echo '<div class="noptin-popup-backdrop"></div>';
	}

	/**
	 * Returns a list of all published popup forms
	 *
	 * @access      public
	 * @since       1.0.5
	 * @return      array
	 */
	public function get_popups() {

		return get_posts(
			array(
				'numberposts' => -1,
				'fields'      => 'ids',
				'post_type'   => 'noptin-form',
				'post_status' => 'publish',
				'meta_query'  => array(
					array(
						'key'     => 'form_settings',
						'compare' => 'EXISTS',
					),
				),
			)
		);

	}

	/**
	 * Retrieves popup forms.
	 *
	 * @access      public
	 * @since       1.0.5
	 * @return      Noptin_Form[]
	 */
	public function get_forms() {

		if ( ! empty( $_GET['interim-login'] ) ) {
			return array();
		}

		// Maybe abort early.
		if ( is_admin() || is_noptin_actions_page() || ! noptin_should_show_optins() || noptin_is_preview() ) {
			return array();
		}

		// Search db once.
		if ( null !== $this->forms ) {
			return $this->forms;
		}

		$this->forms = array();
		$popup_types = array( 'popup', 'slide' );

		foreach ( $this->get_popups() as $popup ) {

			// Prepare the form.
			$form = noptin_get_optin_form( $popup );

			// Can it be displayed?
			if ( ! $form->can_show() ) {
				continue;
			}

			// Loop through each popup type.
			$can_show = false;
			foreach ( $popup_types as $type ) {

				// Is it enabled.
				if ( ! empty( $form->settings[ $type ]['enable'] ) ) {
					$can_show = true;
				}
			}

			if ( $can_show ) {
				$this->forms[] = $form;
			}
		}

		return $this->forms;
	}

	/**
	 * Retrieves popup forms.
	 *
	 * @access      public
	 * @since       1.0.5
	 * @return      Noptin_Form[]
	 */
	public function load_scripts() {

		if ( 0 === count( $this->get_forms() ) ) {
			return;
		}

		// Scripts.
		wp_enqueue_script(
			'noptin-popups',
			noptin()->plugin_url . 'includes/assets/js/dist/popups.js',
			array(),
			filemtime( noptin()->plugin_path . 'includes/assets/js/dist/popups.js' ),
			true
		);

		// Styles.
		wp_enqueue_style(
			'noptin-popups',
			plugin_dir_url( __FILE__ ) . 'noptin-popups.css',
			array(),
			filemtime( plugin_dir_path( __FILE__ ) . 'noptin-popups.css' )
		);

	}

}

new Noptin_AP_Form_Enhancements();
