<?php

namespace Noptin\Addons_Pack\Coupon_Codes;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Fired when a coupon is used.
 *
 * @since 1.0.2
 */
class Used_Coupon_Trigger extends \Noptin_Abstract_Trigger {

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function __construct() {
		add_action( 'woocommerce_order_status_completed', array( $this, 'maybe_trigger_wc' ) );
		add_action( 'edd_complete_purchase', array( $this, 'maybe_trigger_edd' ) );
	}

	/**
	 * @inheritdoc
	 */
	public function get_id() {
		return 'generated-coupon-used';
	}

	/**
	 * @inheritdoc
	 */
	public function get_name() {
		return __( 'Generated Coupon Used', 'noptin-addons-pack' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_description() {
		return __( 'When a generated coupon code is used', 'noptin-addons-pack' );
	}

	/**
	 * Checks if a coupon code is predefined.
	 *
	 * @since 1.0.2
	 * @param string $coupon_code The coupon code.
	 * @return bool
	 */
	public function is_predefined_coupon_code( $coupon_code ) {
		$predefined = noptin_parse_list( strtolower( get_noptin_option( 'coupon_codes_presaved', '' ) ) );

		return in_array( strtolower( $coupon_code ), $predefined, true );
	}

	/**
	 * Triggers for WooCommerce.
	 *
	 * @param int $order_id The order id.
	 */
	public function maybe_trigger_wc( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order ) {
			return;
		}

		$customer = new \WC_Customer( $order->get_customer_id() );

		if ( $customer->get_email() ) {
			$email = $customer->get_email();
		} else {
			$email = $order->get_billing_email();
		}

		foreach ( $order->get_coupon_codes() as $coupon_code ) {
			$coupon = new \WC_Coupon( $coupon_code );

			if ( ! $coupon->get_id() ) {
				continue;
			}

			$is_noptin = $coupon->get_meta( 'is_noptin_coupon' );

			if ( ! $is_noptin && ! $this->is_predefined_coupon_code( $coupon_code ) ) {
				continue;
			}

			$this->trigger(
				$email,
				array(
					'coupon_code' => $coupon_code,
					'email'       => $email,
				)
			);
		}
	}

	/**
	 * Triggers for EDD.
	 *
	 * @param int $payment_id The payment id.
	 */
	public function maybe_trigger_edd( $order_id ) {
		$payment = edd_get_payment( $order_id );

		if ( ! $payment ) {
			return;
		}

		$customer = edd_get_customer( $payment->customer_id );

		if ( $customer ) {
			$email = $customer->email;
		} else {
			$email = $payment->email;
		}

		foreach ( noptin_parse_list( $payment->discounts, true ) as $discount_code ) {
			$discount = new \EDD_Discount( $discount_code, true );

			if ( ! $discount->id ) {
				continue;
			}

			$is_noptin = get_post_meta( $discount->id, 'is_noptin_coupon', true );

			if ( ! $is_noptin && ! $this->is_predefined_coupon_code( $discount_code ) ) {
				continue;
			}

			$this->trigger(
				$email,
				array(
					'coupon_code' => $discount_code,
					'email'       => $email,
				)
			);
		}
	}

	/**
	 * Prepares email test data.
	 *
	 * @since 1.9.0
	 * @param \Noptin_Automation_Rule $rule
	 * @return \Noptin_Automation_Rules_Smart_Tags
	 * @throws \Exception
	 */
	public function get_test_smart_tags( $rule ) {

		$args = $this->prepare_trigger_args(
			'example@gmail.com',
			array(
				'coupon_code' => 'UNIQUE_COUPON_CODE',
				'email'       => 'example@gmail.com',
			)
		);

		return $args['smart_tags'];
	}

}
