<?php

namespace Noptin\Addons_Pack\Users;

defined( 'ABSPATH' ) || exit;

/**
 * Manages user custom fields.
 */
class Custom_Fields {

	/**
	 * Known user custom fields.
	 *
	 * @var array
	 */
	protected static $custom_fields = null;

	/**
	 * Returns a list of known user custom fields.
	 *
	 * @return array
	 */
	public static function get() {

		if ( is_null( self::$custom_fields ) ) {

			$custom_fields = array_merge(
				self::get_acf_fields(),
				self::get_jetengine_fields()
			);

			self::$custom_fields = apply_filters( 'noptin_users_known_custom_fields', $custom_fields );

		}

		return self::$custom_fields;

	}

	/**
	 * Filters user conditional logic rules.
	 *
	 * @param array $rules The rules to filter.
	 * @param Noptin_Abstract_Trigger $trigger The trigger.
	 * @return array
	 */
	public static function filter_conditional_logic_rules( $rules, $trigger ) {

		if ( ! $trigger->is_user_based ) {
			return $rules;
		}

		$custom_fields = self::get();

		if ( empty( $custom_fields ) ) {
			return $rules;
		}

		foreach ( $custom_fields as $field_name => $field ) {

			$rules[ 'user_cf_' . $field_name ] = array(
				'description'       => $field['label'],
				'conditional_logic' => $field['type'],
			);

			if ( isset( $field['options'] ) ) {
				$rules[ 'user_cf_' . $field_name ]['options'] = $field['options'];
			}
		}

		return $rules;

	}

	/**
	 * Returns an array of jetengine fields.
	 *
	 * @return array
	 */
	private static function get_jetengine_fields() {

		if ( ! function_exists( 'jet_engine' ) ) {
			return array();
		}

		$user_field_groups = jet_engine()->meta_boxes->get_fields_for_context( 'user' );
		$user_fields       = array();

		foreach ( $user_field_groups as $fields ) {
			foreach ( $fields as $field ) {
				if ( ! empty( $field['name'] ) && ! empty( $field['object_type'] ) && 'field' === $field['object_type'] && ! in_array( $field['type'], array( 'html' ), true ) ) {
					$user_fields[ $field['name'] ] = array(
						'label' => $field['title'],
						'type'  => 'string',
					);
				}
			}
		}

		return $user_fields;

	}

	/**
	 * Returns an array of ACF fields.
	 *
	 * @return array
	 */
	private static function get_acf_fields() {

		if ( ! function_exists( 'acf_get_field_groups' ) ) {
			return array();
		}

		$user_field_groups = acf_get_field_groups();
		$user_fields       = array();

		if ( ! $user_field_groups ) {
			return array();
		}

		foreach ( $user_field_groups as $field_group ) {

			if ( self::acf_user_has_field_group( $field_group ) ) {
				$user_fields = array_merge( $user_fields, self::acf_flatten_fields( acf_get_fields( $field_group ) ) );
			}
		}

		return $user_fields;

	}

	/**
	 * Gets an array of the location types that a field group is configured to use.
	 *
	 * @param array  $field_group The field group to check.
	 *
	 * @return bool
	 */
	private static function acf_user_has_field_group( $field_group ) {
		if ( empty( $field_group['active'] ) || empty( $field_group['location'] ) || ! is_array( $field_group['location'] ) ) {
			return false;
		}

		$location_types = acf_get_location_types();

		foreach ( $field_group['location'] as $rule_group ) {

			foreach ( $rule_group as $rule ) {
				$rule = acf_validate_location_rule( $rule );

				if ( isset( $location_types[ $rule['param'] ] ) && $location_types[ $rule['param'] ]->object_type && 'user' === $location_types[ $rule['param'] ]->object_type ) {
					return true;
				}
			}
		}

		return false;
	}

	/**
	 * Flattens ACF fields.
	 *
	 * @param array $fields The fields to flatten.
	 * @param string $prefix The prefix to use.
	 * @param string $parent_name The parent name.
	 *
	 * @return bool
	 */
	private static function acf_flatten_fields( $fields, $prefix = '', $parent_name = '' ) {

		$flattened = array();

		foreach ( $fields as $field ) {

			// Skip fields without a name or label.
			if ( empty( $field['name'] ) || empty( $field['label'] ) ) {
				continue;
			}

			$field_name = empty( $prefix ) ? $field['name'] : $prefix . '_' . $field['name'];
			$label      = empty( $parent_name ) ? $field['label'] : $parent_name . ' - ' . $field['label'];

			// Check for sub fields.
			if ( ! empty( $field['sub_fields'] ) ) {
				$flattened = array_merge( $flattened, self::acf_flatten_fields( $field['sub_fields'], $field_name, $label ) );
				continue;
			}

			// Skip presentation fields.
			if ( in_array( $field['type'], array( 'tab', 'accordion', 'message', 'clone' ), true ) ) {
				continue;
			}

			$flattened[ $field_name ] = array(
				'label' => $label,
				'type'  => 'string',
			);

			// Check for choices.
			if ( ! empty( $field['choices'] ) ) {
				$flattened[ $field_name ]['options'] = $field['choices'];
			}

			// Check for numeric fields.
			if ( in_array( $field['type'], array( 'number', 'range' ), true ) ) {
				$flattened[ $field_name ]['type'] = 'number';
			}
		}

		return $flattened;
	}

}
