<?php

namespace Noptin\Addons_Pack\Webhooks;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Registers an automation rule that sends a webhook.
 *
 * @since 1.0.0
 */
class Action extends \Noptin_Abstract_Action {

	/**
	 * @inheritdoc
	 */
	public function get_image() {
		return array(
			'icon' => 'admin-site-alt3',
			'fill' => '#00838f',
		);
	}

	/**
	 * @inheritdoc
	 */
	public function get_id() {
		return 'webhook';
	}

	/**
	 * @inheritdoc
	 */
	public function get_name() {
		return __( 'Send Webhook', 'noptin-addons-pack' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_description() {
		return __( 'Send a webhook', 'noptin-addons-pack' );
	}

	/**
	 * Retrieve the actions's rule table description.
	 *
	 * @since 1.11.9
	 * @param Noptin_Automation_Rule $rule
	 * @return array
	 */
	public function get_rule_table_description( $rule ) {

		$settings = $rule->action_settings;

		// Ensure we have a field name.
		if ( empty( $settings['url'] ) ) {
			return sprintf(
				'<span class="noptin-rule-error">%s</span>',
				esc_html__( 'Error: No webhook URL specified', 'noptin-addons-pack' )
			);
		}

		return $this->rule_action_meta(
			array(
				__( 'Webhook URL', 'noptin-addons-pack' ) => esc_url( $settings['url'] ),
				__( 'Method', 'noptin-addons-pack' )      => empty( $settings['method'] ) ? 'POST' : esc_html( $settings['method'] ),
			),
			$rule
		);
	}

	/**
	 * @inheritdoc
	 */
	public function get_settings() {

		return array(

			'url'          => array(
				'el'          => 'input',
				'type'        => 'text',
				'label'       => __( 'Webhook URL', 'noptin-addons-pack' ),
				'placeholder' => __( 'https://hooks.zapier.com/hooks/catch/123456/a1b23c4/', 'noptin-addons-pack' ),
				'description' => __( 'The URL to send the webhook to.', 'noptin-addons-pack' ),
				'default'     => '',
			),

			'method'       => array(
				'el'          => 'select',
				'label'       => __( 'Webhook Method', 'noptin-addons-pack' ),
				'description' => __( 'The method of the webhook.', 'noptin-addons-pack' ),
				'options'     => array(
					'GET'  => __( 'GET', 'noptin-addons-pack' ),
					'POST' => __( 'POST', 'noptin-addons-pack' ),
					'PUT'  => __( 'PUT', 'noptin-addons-pack' ),
				),
				'default'     => 'POST',
			),

			'content_type' => array(
				'el'          => 'select',
				'label'       => __( 'Webhook Content Type', 'noptin-addons-pack' ),
				'description' => __( 'The content type of the webhook.', 'noptin-addons-pack' ),
				'options'     => array(
					'json' => __( 'JSON', 'noptin-addons-pack' ),
					'form' => __( 'Form Data', 'noptin-addons-pack' ),
				),
				'default'     => 'form',
				'restrict'    => "action_settings.method !== 'GET'",
			),

			'content'      => array(
				'el'          => 'webhook_key_value_repeater',
				'label'       => __( 'Data', 'noptin-addons-pack' ),
				'description' => __( 'Enter the content to send with the webhook or leave blank to send everything.', 'noptin-addons-pack' ),
				'default'     => array(),
			),

			'headers'      => array(
				'el'          => 'webhook_key_value_repeater',
				'label'       => __( 'Headers', 'noptin-addons-pack' ),
				'description' => __( 'Enter the headers to send as part of the webhook.', 'noptin-addons-pack' ),
				'default'     => array(
					array(
						'key'   => 'X-Origin',
						'value' => 'Noptin',
					),
				),
			),
		);
	}

	/**
	 * Sends a webhook.
	 *
	 * @since 1.0.0
	 * @param mixed $subject The subject.
	 * @param Noptin_Automation_Rule $rule The automation rule that triggered the action.
	 * @param array $args Extra arguments passed to the action.
	 * @return void
	 */
	public function run( $subject, $rule, $_args ) {

		$settings = wp_unslash( $rule->action_settings );

		/** @var Noptin_Automation_Rules_Smart_Tags $smart_tags */
		$smart_tags = $_args['smart_tags'];

		// Abort if no URL is set.
		if ( empty( $settings['url'] ) ) {
			return;
		}

		// Main URL.
		$url = esc_url( $smart_tags->replace_in_text_field( $settings['url'] ) );

		// Content.
		$content = array();

		if ( ! empty( $settings['content'] ) ) {
			foreach ( $settings['content'] as $item ) {
				$content[ $item['key'] ] = $smart_tags->replace_in_text_field( $item['value'] );
			}
		} else {
			foreach ( $smart_tags->tags as $key => $config ) {

				if ( 'cookie' === $key ) {
					continue;
				}

				if ( ! empty( $config['example'] ) ) {
					$tag = $config['example'];
				} else {
					$tag = $key;
				}

				$content[ $key ] = $smart_tags->replace_in_text_field( '[[' . $tag . ']]' );
			}
		}

		// Headers.
		$headers = array( 'X-Rule-ID' => $rule->id );

		if ( ! empty( $settings['headers'] ) ) {
			foreach ( $settings['headers'] as $item ) {
				$headers[ $item['key'] ] = $smart_tags->replace_in_text_field( $item['value'] );
			}
		}

		// Method.
		$method = strtoupper( $settings['method'] );

		// Content type.
		if ( 'GET' !== $method ) {
			if ( 'json' === $settings['content_type'] ) {
				$content                 = wp_json_encode( $content );
				$headers['Content-Type'] = 'application/json';
			} else {
				$headers['Content-Type'] = 'application/x-www-form-urlencoded';
			}
		} else {
			$url     = add_query_arg( urlencode_deep( $content ), $url );
			$content = false;
		}

		// Filter the webhook data.
		$request = apply_filters(
			'noptin_automation_webhook_request',
			array(
				'url'     => $url,
				'method'  => $method,
				'headers' => $headers,
				'body'    => $content,
			),
			$rule,
			$subject,
			$_args
		);

		if ( false === $request['body'] ) {
			unset( $request['body'] );
		}

		// Send the webhook.
		wp_remote_request( $request['url'], $request );
	}

}
