<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * License Manager
 *
 * @since 5.3.2
 */
class WCSED_License_Manager {

    /**
     * Property that holds the single main instance of WCSED_License_Manager.
     *
     * @access private
     * @since 5.3.2
     * @var WCSED_License_Manager
     */
    private static $_instance;

    /**
     * Property that holds the slug of the license settings page.
     *
     * @since 5.3.2
     * @access private
     * @var string
     */
    private $_settings_page_slug;

    /**
     * Property that holds the software key of this plugin.
     *
     * @since 5.3.2
     * @access private
     * @var string $_software_key
     */
    private $_software_key;


    /**
     * Property that holds the activation endpoint.
     *
     * @since 5.3.2
     * @access private
     * @var string
     */
    private $_activation_endpoint;

    /**
     * Class Methods
     */

    /**
     * WCSED_License_Manager constructor.
     *
     * @access public
     * @since 5.3.2
     */
    public function __construct() {
        $this->_software_key        = 'SED';
        $this->_settings_page_slug  = 'visserlabs_license_settings';
        $this->_activation_endpoint = 'https://visser.com.au/wp-json/slmw/v1/license/activate';

        if ( is_multisite() && get_current_blog_id() === 1 ) {

            // Add Visser Labs license settings menu in multi-site environment.
            add_action( 'network_admin_menu', array( $this, 'register_ms_wcsed_licenses_settings_menu' ), 10 );

            // Add license activation notice in multi-site environment.
            add_action( 'network_admin_notices', array( $this, 'activate_license_notice' ) );

        } else {

            // Add Add Visser Labs license settings menu.
            add_action( 'admin_menu', array( $this, 'register_wcsed_license_settings_menu' ), 10 );

            // Add license activation notice.
            add_action( 'admin_notices', array( $this, 'activate_license_notice' ) );

        }

        // Enqueue license settings page scripts and styles.
        add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_wcsed_license_settings_page_scripts' ), 10 );

        // Load the license settings page content.
        add_action( 'vl_license_settings_page_content', array( $this, 'load_wcsed_license_settings_page_content' ), 10 );

        // Activate license.
        add_action( 'wp_ajax_wcsed_activate_license', array( $this, 'wcsed_activate_license' ) );

        // Dismiss activate license notice.
        add_action( 'wp_ajax_wcsed_slmw_dismiss_activate_notice', array( $this, 'ajax_dismiss_activate_notice' ) );
    }

    /**
     * Ensure that only one instance of WCSED_License_Manager is loaded or can be loaded (Singleton Pattern).
     *
     * @param array $dependencies Array of instance objects of all dependencies of WCSED_License_Manager model.
     *
     * @return WCSED_License_Manager
     * @since 5.3.2
     */
    public static function instance( $dependencies = null ) {

        if ( ! self::$_instance instanceof self ) {
            self::$_instance = new self( $dependencies );
        }

        return self::$_instance;
    }

    /**
     * Register Visser Labs license settings menu.
     *
     * @access public
     * @since 5.3.2
     */
    public function register_wcsed_license_settings_menu() {
        if ( ! defined( 'VISSERLABS_LICENSE_SETTINGS_PAGE' ) ) {

            if ( ! defined( 'VISSERLABS_LICENSE_SETTINGS_DEFAULT_PLUGIN' ) ) {
                define( 'VISSERLABS_LICENSE_SETTINGS_DEFAULT_PLUGIN', 'wcsed' );
            }

            add_submenu_page(
                'options-general.php',
                __( 'Visser Labs License', 'woocommerce-exporter' ),
                __( 'Visser Labs License', 'woocommerce-exporter' ),
                'manage_woocommerce',
                $this->_settings_page_slug,
                array( $this, 'wcsed_license_settings_page' )
            );

            define( 'VISSERLABS_LICENSE_SETTINGS_PAGE', 'store-exporter' );
        }
    }

    /**
     * Register Visser Labs license settings menu in multi-site environment.
     *
     * @access public
     * @since 5.3.2
     */
    public function register_ms_wcsed_licenses_settings_menu() {
        if ( ! defined( 'VISSERLABS_LICENSE_SETTINGS_PAGE' ) ) {

            if ( ! defined( 'VISSERLABS_LICENSE_SETTINGS_DEFAULT_PLUGIN' ) ) {
                define( 'VISSERLABS_LICENSE_SETTINGS_DEFAULT_PLUGIN', 'wcsed' );
            }

            add_menu_page(
                __( 'Visser Labs License', 'woocommerce-exporter' ),
                __( 'Visser Labs License', 'woocommerce-exporter' ),
                'manage_sites',
                $this->_settings_page_slug,
                array( $this, 'wcsed_license_settings_page' )
            );

            define( 'VISSERLABS_LICENSE_SETTINGS_PAGE', 'store-exporter' );
        }
    }

    /**
     * Load the license settings page.
     *
     * @access public
     * @since 5.3.2
     */
    public function wcsed_license_settings_page() {
        $tab = filter_input( INPUT_GET, 'tab', FILTER_UNSAFE_RAW ) ?? '';
        $tab = htmlspecialchars( $tab, ENT_QUOTES, 'UTF-8' );
        $tab = empty( $tab ) ? 'store-exporter' : $tab;

        $vl_plugins_tabs = array(
            array(
                'id'     => 'store-exporter',
                'title'  => __( 'Store Exporter', 'woocommerce-exporter' ),
                'url'    =>
                    is_multisite() ?
                        network_admin_url( 'admin.php?page=' . $this->_settings_page_slug . '&tab=store-exporter' ) :
                        admin_url( 'options-general.php?page=' . $this->_settings_page_slug . '&tab=store-exporter' ),
                'active' => 'store-exporter' === $tab ?? true,
            ),
        );

        if ( $this->_is_product_importer_active() || $this->_is_product_importer_deluxe_active() ) {
            $vl_plugins_tabs[] = array(
                'id'     => 'product-importer',
                'title'  => __( 'Product Importer', 'woocommerce-exporter' ),
                'url'    => is_multisite() ?
                    network_admin_url( 'admin.php?page=' . $this->_settings_page_slug . '&tab=product-importer' ) :
                    admin_url( 'options-general.php?page=' . $this->_settings_page_slug . '&tab=product-importer' ),
                'active' => 'product-importer' === $tab ?? true,
            );
        }

        require_once WOO_CD_PATH . 'templates/admin/license-settings-page.php';
    }

    /**
     * Enqueue license settings page scripts and styles.
     *
     * @access public
     * @since 5.3.2
     */
    public function enqueue_wcsed_license_settings_page_scripts() {
        $page = filter_input( INPUT_GET, 'page', FILTER_UNSAFE_RAW ) ?? '';
        $page = htmlspecialchars( $page, ENT_QUOTES, 'UTF-8' );

        if ( $this->_settings_page_slug === $page ) {
            wp_enqueue_style( 'wcsed-license-settings-style', plugins_url( '/css/license-settings.css', WOO_CD_RELPATH ), array(), WOO_CD_VERSION );
            wp_enqueue_script( 'wcsed-license-settings-script', plugins_url( '/js/license-settings.js', WOO_CD_RELPATH ), array( 'jquery', 'jquery-ui-tabs' ), WOO_CD_VERSION, true );
            wp_localize_script(
                'wcsed-license-settings-script',
                'wcsed_license_settings_args',
                array(
                    'i18n' => array(
                        'inactive' => __( 'Inactive', 'woocommerce-exporter' ),
                        'active'   => __( 'Active', 'woocommerce-exporter' ),
                        'expired'  => __( 'Expired', 'woocommerce-exporter' ),
                    ),
                )
            );
        }
    }

    /**
     * Load the license settings page content.
     *
     * @access public
     * @since 5.3.2
     */
    public function load_wcsed_license_settings_page_content() {
        $tab = filter_input( INPUT_GET, 'tab', FILTER_UNSAFE_RAW ) ?? '';
        $tab = htmlspecialchars( $tab, ENT_QUOTES, 'UTF-8' );

        if ( empty( $tab ) || 'store-exporter' === $tab ) {
            $wcsed_activation_email          = is_multisite() ? get_site_option( 'wcsed_option_activation_email' ) : get_option( 'wcsed_option_activation_email' );
            $wcsed_license_key               = is_multisite() ? get_site_option( 'wcsed_option_license_key' ) : get_option( 'wcsed_option_license_key' );
            $wcsed_license_status_value      = is_multisite() ? get_site_option( 'wcsed_license_status' ) : get_option( 'wcsed_license_status' );
            $wcsed_license_expired_timestamp = is_multisite() ? get_site_option( 'wcsed_license_expired' ) : get_option( 'wcsed_license_expired' );
            $wcsed_license_expired_date      = ! empty( $wcsed_license_expired_timestamp ) ? wp_date( get_option( 'date_format' ), $wcsed_license_expired_timestamp ) : '';

            $wcsed_license_status = array(
                'inactive' => array(
                    'text'  => __( 'Inactive', 'woocommerce-exporter' ),
                    'class' => 'text-color-red',
                ),
                'active'   => array(
                    'text'  => __( 'Active', 'woocommerce-exporter' ),
                    'class' => 'text-color-green',
                ),
                'expired'  => array(
                    'text'  => __( 'Expired', 'woocommerce-exporter' ) . ' <small><i>(' . $wcsed_license_expired_date . ')</i></small>',
                    'class' => 'text-color-red',
                ),
            );

            $wcsed_license_status_i18n       = isset( $wcsed_license_status[ $wcsed_license_status_value ] ) ? $wcsed_license_status[ $wcsed_license_status_value ]['text'] : $wcsed_license_status['inactive']['text'];
            $wcsed_license_status_class_name = isset( $wcsed_license_status[ $wcsed_license_status_value ] ) ? $wcsed_license_status[ $wcsed_license_status_value ]['class'] : $wcsed_license_status['inactive']['class'];

            include_once WOO_CD_PATH . 'templates/admin/license-settings-store-exporter-page.php';
        }
    }

    /**
     * Activate license.
     *
     * @access public
     * @since 5.3.2
     */
    public function wcsed_activate_license() {
        if ( ! defined( 'DOING_AJAX' ) || ! DOING_AJAX ) {
            $response = array(
				'status'    => 'fail',
				'error_msg' => __( 'Invalid AJAX Operation', 'woocommerce-exporter' ),
			);
        } elseif ( ! isset( $_POST['activation_email'] ) || ! isset( $_POST['license_key'] ) || ! isset( $_POST['ajax_nonce'] ) ) {
            $response = array(
				'status'    => 'fail',
				'error_msg' => __( 'Required parameters not supplied', 'woocommerce-exporter' ),
			);
        } elseif ( ! check_ajax_referer( 'wcsed_activate_license_nonce', 'ajax_nonce', false ) ) {
            $response = array(
				'status'    => 'fail',
				'error_msg' => __( 'Security check failed', 'woocommerce-exporter' ),
			);
        } else {
            $activation_email = trim( sanitize_text_field( wp_unslash( $_POST['activation_email'] ?? '' ) ) );
            $license_key      = trim( sanitize_text_field( wp_unslash( $_POST['license_key'] ?? '' ) ) );

            $activation_url = add_query_arg(
                array(
                    'activation_email' => rawurlencode( $activation_email ),
                    'license_key'      => $license_key,
                    'site_url'         => home_url(),
                    'software_key'     => $this->_software_key,
                    'multisite'        => is_multisite() ? 1 : 0,
                ),
                apply_filters( 'wcsed_license_activation_url', $this->_activation_endpoint )
            );

            // Store data even if not valid license.
            if ( is_multisite() ) {
                update_site_option( 'wcsed_option_license_key', $license_key );
                update_site_option( 'wcsed_option_activation_email', $activation_email );
            } else {
                update_option( 'wcsed_option_license_key', $license_key );
                update_option( 'wcsed_option_activation_email', $activation_email );
            }

            $option = apply_filters(
                'wcsed_license_activation_options',
                array(
                    'timeout' => 10, // Seconds.
                    'headers' => array( 'Accept' => 'application/json' ),
                )
            );

            $result = wp_remote_retrieve_body( wp_remote_get( $activation_url, $option ) );

            // Activation point failed to send a response.
            if ( empty( $result ) ) {
                $response = array(
                    'status'    => 'fail',
                    'error_msg' => __( 'Failed to activate license. Failed to connect to activation access point. Please contact plugin support.', 'woocommerce-exporter' ),
                );
            } else {
                $result = json_decode( $result );

                if ( empty( $result ) || ! property_exists( $result, 'status' ) ) {
                    $response = array(
                        'status'  => 'fail',
                        'message' => __( 'Failed to activate license. Activation access point return invalid response. Please contact plugin support.', 'advanced-coupons-for-woocommerce' ),
                    );
                } elseif ( 'success' === $result->status ) {
                    if ( is_multisite() ) {
                        update_site_option( 'wcsed_license_activated', 'yes' );
                    } else {
                        update_option( 'wcsed_license_activated', 'yes' );
                    }

                    $response = array(
                        'status'  => $result->status,
                        'message' => ! empty( $result->error_msg ) ? $result->error_msg : $result->success_msg,
                    );
                } else {
                    if ( is_multisite() ) {
                        update_site_option( 'wcsed_license_activated', 'no' );
                    } else {
                        update_option( 'wcsed_license_activated', 'no' );
                    }

                    $response = array(
                        'status'  => $result->status,
                        'message' => ! empty( $result->error_msg ) ? $result->error_msg : $result->message,
                    );
                }
            }

            // set license status based on response.
            $license_status = $this->_get_license_status_based_on_slmw_response( (array) $result );

            // update license status site option.
            if ( is_multisite() ) {
                update_site_option( 'wcsed_license_status', $license_status );
            } else {
                update_option( 'wcsed_license_status', $license_status );
            }

            // Check if this license is expired.
            if ( 'expired' === $license_status ) {
                // Always store in GMT in database.
                $response['expired_date'] = wp_date( get_option( 'date_format' ), $result->expiration_timestamp );

                if ( is_multisite() ) {
                    update_site_option( 'wcsed_license_expired', $result->expiration_timestamp );
                } else {
                    update_option( 'wcsed_license_expired', $result->expiration_timestamp );
                }
            } elseif ( is_multisite() ) {
                    delete_site_option( 'wcsed_license_expired' );
                } else {
                    delete_option( 'wcsed_license_expired' );
            }

            $response['license_status'] = $license_status;
        }

        // if license status is not set, then we set it as empty value.
        if ( ! isset( $response['license_status'] ) ) {
            $response['license_status'] = '';
        }

        @header( 'Content-Type: application/json; charset=' . get_option( 'blog_charset' ) ); // phpcs:ignore
        echo wp_json_encode( $response );
        wp_die();
    }

    /**
     * Get license status based on SLMW response
     *
     * @since 5.3.2
     * @access private
     *
     * @param array $result SLMW response.
     * @return string License status.
     */
    private function _get_license_status_based_on_slmw_response( $result ) {
        // return empty if the response data has no status prop.
        if ( ! is_array( $result ) || ! isset( $result['status'] ) ) {
            return '';
        }

        // handle failed response.
        if ( 'fail' === $result['status'] ) {

            // return as expired when the expiration keys are present.
            if ( isset( $result['expiration_timestamp'] ) || isset( $result['expired_date'] ) ) {
                return 'expired';
            }

            return 'inactive';
        }

        return 'active';
    }

    /**
     * Activate license notice.
     *
     * @since 5.3.2
     * @access public
     */
    public function activate_license_notice() {

        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        if ( is_multisite() ) {

            $license_activated = get_site_option( 'wcsed_license_activated' );
            $notice_dismissed  = get_site_option( 'wcsed_slmw_active_notice_dismissed' );

        } else {

            $license_activated = get_option( 'wcsed_license_activated' );
            $notice_dismissed  = get_option( 'wcsed_slmw_active_notice_dismissed' );

        }

        if ( 'yes' !== $license_activated && 'yes' !== $notice_dismissed ) {

            global $wp;

            if ( is_multisite() ) {
                $wcsed_slmw_settings_url = network_admin_url( 'admin.php?page=' . $this->_settings_page_slug . '&tab=store-exporter' );
            } else {
                $wcsed_slmw_settings_url = admin_url( 'options-general.php?page=' . $this->_settings_page_slug . '&tab=store-exporter' );
            }

            ?>

            <div class="notice notice-error is-dismissible wcsed-activate-license-notice">
                <h4>
                    <?php
                    echo wp_kses_post(
                        sprintf(
                            /* Translators: %1$s: Link tag start. %2$s: Link tag end. */
                            __( 'Please %1$sactivate%2$s your copy of WooCommerce Store Exporter license to get the latest updates and have access to support.', 'woocommerce-exporter' ),
                            '<a href="' . $wcsed_slmw_settings_url . '">',
                            '</a>'
                        )
                    );
                        ?>
                </h4>
            </div>
            <style>.wcsed-activate-license-notice .notice-dismiss { margin-top: 8px; }</style>
            <script>
                jQuery( document ).ready( function($){
                    $( '.wcsed-activate-license-notice' ).on( 'click' , '.notice-dismiss' , function() {
                        console.log( 'test' );
                        $.post( window.ajaxurl, { action : 'wcsed_slmw_dismiss_activate_notice' } );
                    } );
                });
            </script>

        <?php
        }
    }

    /**
     * AJAX dismiss activate notice.
     *
     * @since 5.3.2
     * @access public
     */
    public function ajax_dismiss_activate_notice() {

        if ( ! defined( 'DOING_AJAX' ) || ! DOING_AJAX ) {
            $response = array(
				'status'    => 'fail',
				'error_msg' => __( 'Invalid AJAX Operation', 'woocommerce-exporter' ),
			);
        } else {

            if ( is_multisite() ) {
                update_site_option( 'wcsed_slmw_active_notice_dismissed', 'yes' );
            } else {
                update_option( 'wcsed_slmw_active_notice_dismissed', 'yes' );
            }

            $response = array( 'status' => 'success' );

        }

        @header( 'Content-Type: application/json; charset=' . get_option( 'blog_charset' ) ); // phpcs:ignore
        echo wp_json_encode( $response );
        wp_die();
    }

    /**
     * Check if Product Importer is active.
     *
     * @access private
     * @since 5.3.2
     *
     * @return bool
     */
    private function _is_product_importer_active() {
        return is_plugin_active( 'woocommerce-product-importer/product-importer.php' );
    }

    /**
     * Check if Product Importer Deluxe is active.
     *
     * @access private
     * @since 5.3.2
     *
     * @return bool
     */
    private function _is_product_importer_deluxe_active() {
        return is_plugin_active( 'woocommerce-product-importer-deluxe/product-importer-deluxe.php' );
    }
}
