<?php

namespace Noptin\WooCommerce;

/**
 * Connects Noptin to WooCommerce.
 *
 * @version 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * The main plugin class.
 */
class Plugin {

	/**
	 * @var \Noptin\WooCommerce\Subscriptions\Main
	 */
	public $subscriptions;

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'admin_init', array( $this, 'maybe_schedule_background_action' ) );
		add_action( 'noptin_automation_rules_load', array( $this, 'load_automation_rules' ) );
	}

	/**
	 * Schedules the background action if it's not already scheduled.
	 */
	public function maybe_schedule_background_action() {
		if ( function_exists( 'as_next_scheduled_action' ) && false === as_next_scheduled_action( 'noptin_woocommerce_daily_action', null, 'noptin' ) ) {
			as_schedule_recurring_action( strtotime( 'midnight tonight' ), DAY_IN_SECONDS, 'noptin_woocommerce_daily_action', array(), 'noptin' );
		}
	}

	/**
	 * Loads automation rules.
	 * @param \Noptin_Automation_Rules $rules
	 */
	public function load_automation_rules( $rules ) {

		$rules->add_trigger( new Triggers\Customer_Last_Order_Date() );

		// Check if wc subscriptions is active.
		if ( class_exists( 'WC_Subscriptions' ) ) {

			foreach ( $this->get_subscription_status_hooks() as $hook => $details ) {
				$rules->add_trigger( new Triggers\Subscription_Status_Change( $hook, $details['name'], $details['description'] ) );
			}

			$rules->add_trigger( new Triggers\Subscription_Before_End() );
			$rules->add_trigger( new Triggers\Subscription_Before_Renewal() );
			$rules->add_trigger( new Triggers\Saved_Card_Before_Expiry() );

			$this->subscriptions = new Subscriptions\Main();
		}
	}

	/**
	 * Returns subscription status hooks.
	 *
	 * @return array
	 */
	public function get_subscription_status_hooks() {
		return array(
			'woocommerce_scheduled_subscription_trial_end' => array(
				'name'        => __( 'Trial End', 'noptin-woocommerce' ),
				'description' => __( 'When the trial period for a subscription has reached its end date', 'noptin-woocommerce' ),
			),
			'woocommerce_subscription_status_on-hold'      => array(
				'name'        => __( 'Suspended', 'noptin-woocommerce' ),
				'description' => __( 'When a subscription is suspended', 'noptin-woocommerce' ),
			),
			'woocommerce_subscription_renewal_payment_failed' => array(
				'name'        => __( 'Renewal Payment Fails', 'noptin-woocommerce' ),
				'description' => __( "When a subscription's renewal payment fails", 'noptin-woocommerce' ),
			),
			'woocommerce_subscription_renewal_payment_complete' => array(
				'name'        => __( 'Renewal Payment Complete', 'noptin-woocommerce' ),
				'description' => __( "When a subscription's renewal payment completes", 'noptin-woocommerce' ),
			),
			'woocommerce_scheduled_subscription_end_of_prepaid_term' => array(
				'name'        => __( 'End of Prepaid Term', 'noptin-woocommerce' ),
				'description' => __( 'When a subscription that was cancelled by a customer or store owner has reached the end of the term covered by the last payment', 'noptin-woocommerce' ),
			),
			'woocommerce_subscription_status_expired'      => array(
				'name'        => __( 'Expired', 'noptin-woocommerce' ),
				'description' => __( 'When a subscription expires', 'noptin-woocommerce' ),
			),
			'woocommerce_scheduled_subscription_expiration' => array(
				'name'        => __( 'End', 'noptin-woocommerce' ),
				'description' => __( 'When a subscription has reached its end date', 'noptin-woocommerce' ),
			),
			'woocommerce_checkout_subscription_created'    => array(
				'name'        => __( 'Created', 'noptin-woocommerce' ),
				'description' => __( 'When a subscription is created', 'noptin-woocommerce' ),
			),
			'woocommerce_subscription_status_cancelled'    => array(
				'name'        => __( 'Cancelled', 'noptin-woocommerce' ),
				'description' => __( 'When a subscription is cancelled', 'noptin-woocommerce' ),
			),
			'woocommerce_subscription_status_active'       => array(
				'name'        => __( 'Activated', 'noptin-woocommerce' ),
				'description' => __( 'When a subscription is activated', 'noptin-woocommerce' ),
			),
		);
	}

}
