<?php

namespace Noptin\WooCommerce\Triggers;

/**
 * Customer last order date.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Customer last order date.
 *
 * @since 1.0.0
 */
class Customer_Last_Order_Date extends \Noptin_WooCommerce_Trigger {

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function __construct() {
		add_action( 'noptin_woocommerce_daily_action', array( $this, 'maybe_trigger' ) );
		add_action( 'handle_noptin_customer_last_order_trigger', array( $this, 'trigger_for_order' ), 10, 2 );
	}

	/**
	 * @inheritdoc
	 */
	public function get_id() {
		return 'woocommerce_customer_last_order_date';
	}

	/**
	 * @inheritdoc
	 */
	public function get_name() {
		return __( 'WooCommerce Customers > Last Order Date', 'noptin-woocommerce' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_description() {
		return __( "X days since a customer's last order date", 'noptin-woocommerce' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_rule_description( $rule ) {
		$settings = $rule->trigger_settings;

		// Abort if days are not specified.
		if ( empty( $settings['days'] ) ) {
			return $this->get_description();
		}

		return sprintf(
			/* translators: %d days since last order */
			_n( "%d day since a customer's last order", "%d days since a customer's last order", $settings['days'], 'noptin-woocommerce' ),
			(int) $settings['days']
		);
	}

	/**
	 * @inheritdoc
	 */
	public function get_settings() {

		return array_merge(
			array(
				'days' => array(
					'el'          => 'input',
					'type'        => 'number',
					'label'       => __( 'Days since last order', 'noptin-woocommerce' ),
					'description' => __( "Enter the number days since a customer's last order for which this trigger will fire.", 'noptin-woocommerce' ),
					'placeholder' => '200',
				),
			),
			parent::get_settings()
		);

	}

	/**
	 * Fetch customers with a last order date within the specified days.
	 *
	 * @param int $days The days since last order.
	 * @return int[]
	 */
	public function get_orders( $days ) {

		$orders = wc_get_orders(
			array(
				'date_created' => gmdate( 'Y-m-d', strtotime( "-{$days} days" ) ),
				'type'         => 'shop_order',
				'status'       => wc_get_is_paid_statuses(),
				'limit'        => -1,
			)
		);

		$customers = array();

		// Ensure each customer is unique.
		foreach ( $orders as $order ) {

			// Abort if the order is not complete.
			if ( ! $order->is_paid() ) {
				continue;
			}

			if ( ! $order->get_customer_id() ) {
				$customers[ $order->get_billing_email() ] = $order->get_id();
			} else {
				$customers[ $order->get_customer_id() ] = $order->get_id();
			}
		}

		return array_values( $customers );

	}

	/**
     * Returns an array of known smart tags.
     *
     * @since 1.9.0
     * @return array
     */
    public function get_known_smart_tags() {

		return array_merge(
			parent::get_known_smart_tags(),
			$this->get_order_smart_tags(),
			$this->get_customer_smart_tags()
		);
    }

	/**
	 * Maybe triggers an action.
	 */
	public function maybe_trigger() {

		// Loop through all rules.
		foreach ( $this->get_rules() as $rule ) {

            // Prepare the rule.
            $rule = noptin()->automation_rules->prepare_rule( $rule );

			if ( ! isset( $rule->trigger_settings['days'] ) ) {
				continue;
			}

			// Fetch orders.
			$orders = $this->get_orders( intval( $rule->trigger_settings['days'] ) );

			// Abort if no orders are found.
			if ( empty( $orders ) ) {
				continue;
			}

			// Trigger for each order.
			foreach ( $orders as $index => $order ) {

				if ( $index > 10 ) {
					$schedule_for = floor( $index / 10 ) * MINUTE_IN_SECONDS;
					schedule_noptin_background_action( time() + $schedule_for, 'handle_noptin_customer_last_order_trigger', $order, $rule->id );
				} else {
					schedule_noptin_background_action( time() + 30, 'handle_noptin_customer_last_order_trigger', $order, $rule->id );
				}
			}
        }

	}

	/**
	 * Triggers an action for a given order.
	 *
	 * @param \WC_Order $order
	 */
	public function trigger_for_order( $order_id, $rule_id ) {

		$order = wc_get_order( $order_id );

		if ( ! $order || is_wp_error( $order ) || ! $order->is_paid() ) {
			return;
		}

		$customer = new \WC_Customer( $order->get_customer_id() );

		// Set customer data from order if customer is not found.
		if ( ! $customer->get_id() ) {
			$customer->set_email( $order->get_billing_email() );
			$customer->set_billing_email( $order->get_billing_email() );
			$customer->set_first_name( $order->get_billing_first_name() );
			$customer->set_last_name( $order->get_billing_last_name() );
		}

		$args = $this->before_trigger_wc( $order, $customer );

		$args['order']    = $order;
		$args['customer'] = $customer;
		$args['order_id'] = $order->get_id();
		$args['rule_id']  = $rule_id;

		$this->trigger( $customer, $args );

		$this->after_trigger_wc( $args );
	}

	/**
	 * Triggers action callbacks.
	 *
	 * @since 1.2.8
	 * @param \WC_Customer $subject The subject.
	 * @param array $args Extra arguments passed to the action.
	 * @return void
	 */
	public function trigger( $subject, $args ) {

		if ( empty( $args['rule_id'] ) ) {
			return;
		}

		$args = $this->prepare_trigger_args( $subject, $args );
		$rule = new \Noptin_Automation_Rule( $args['rule_id'] );

		// Retrieve the action.
		$action = noptin()->automation_rules->get_action( $rule->action_id );
		if ( empty( $action ) ) {
			return;
		}

		// Set the current email.
		$GLOBALS['current_noptin_email'] = $this->get_subject_email( $subject, $rule, $args );

		// Are we delaying the action?
		$delay = $rule->get_delay();

		if ( $delay > 0 ) {
			do_action( 'noptin_delay_automation_rule_execution', $rule, $args, $delay );
			return;
		}

		// Ensure that the rule is valid for the provided args.
		if ( $this->is_rule_valid_for_args( $rule, $args, $subject, $action ) ) {
			$action->maybe_run( $subject, $rule, $args );
		}

	}

	/**
	 * Serializes the trigger args.
	 *
	 * @since 1.11.1
	 * @param array $args The args.
	 * @return false|array
	 */
	public function serialize_trigger_args( $args ) {
		return array(
			'order_id' => $args['order_id'],
			'rule_id'  => $args['rule_id'],
		);
	}

	/**
	 * Unserializes the trigger args.
	 *
	 * @since 1.11.1
	 * @param array $previous_args The args.
	 * @return array|false
	 */
	public function unserialize_trigger_args( $previous_args ) {

		$order = wc_get_order( $previous_args['order_id'] );

		if ( empty( $order ) || ! is_a( $order, 'WC_Order' ) ) {
			throw new \Exception( 'The order no longer exists' );
		}

		$customer = new \WC_Customer( $order->get_customer_id() );

		if ( ! $customer->get_id() ) {
			$customer->set_email( $order->get_billing_email() );
			$customer->set_billing_email( $order->get_billing_email() );
			$customer->set_first_name( $order->get_billing_first_name() );
			$customer->set_last_name( $order->get_billing_last_name() );
		}

		$args = array_merge(
			$previous_args,
			$this->before_trigger_wc( $order, $customer )
		);

		$args['order']    = $order;
		$args['customer'] = $customer;
		$args['order_id'] = $order->get_id();

		// Prepare the trigger args.
		return $this->prepare_trigger_args( $customer, $args );
	}
}
