<?php

namespace Noptin\WooCommerce\Triggers;

/**
 * Base for all WooCommerce triggers.
 *
 * @since   1.0.0
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Base for all license triggers.
 *
 * @since 1.0.0
 */
abstract class Subscription_Trigger extends \Noptin_WooCommerce_Trigger {

	/**
     * @var \WC_Subscription
     */
	public static $subscription;

	/**
	 * @inheritdoc
	 */
	public $category = 'WooCommerce Subscriptions';

	/**
	 * @inheritdoc
	 */
	public function get_settings() {

		return array_merge(
			array(
				'products' => array(
					'el'          => 'input',
					'label'       => __( 'Products', 'noptin-woocommerce' ),
					'description' => __( 'Enter a comma-separated list of product IDs for which this trigger will fire or leave blank to fire for all products.', 'noptin-woocommerce' ),
					'placeholder' => '100, 101, 102',
				),
			),
			parent::get_settings()
		);

	}

	/**
	 * Checks if subscription has selected products.
	 *
	 * @since 1.0.0
	 * @param \Noptin_Automation_Rule $rule The rule to check for.
	 * @param mixed $args Extra args for the action.
	 * @param mixed $subject The subject.
	 * @param \Noptin_Abstract_Action $action The action to run.
	 * @return bool
	 */
	public function is_rule_valid_for_args( $rule, $args, $subject, $action ) {

		if ( ! empty( $rule->trigger_settings['products'] ) && self::$subscription ) {

			$products = array_filter( wp_parse_id_list( $rule->trigger_settings['products'] ) );

			if ( ! empty( $products ) ) {

				$has_product = false;

				foreach ( $products as $product_id ) {

					if ( self::$subscription->has_product( $product_id ) ) {
						$has_product = true;
						break;
					}
				}

				if ( ! $has_product ) {
					return false;
				}
			}
		}

		return parent::is_rule_valid_for_args( $rule, $args, $subject, $action );
	}
	/**
     * Returns an array of known smart tags.
     *
     * @since 1.9.0
     * @return array
     */
    public function get_known_smart_tags() {

		return array_merge(
			parent::get_known_smart_tags(),
			$this->get_subscription_smart_tags(),
			$this->get_order_smart_tags(),
			$this->get_customer_smart_tags()
		);
    }

	/**
	 * Returns subscription smart tags.
	 *
	 * @return array
	 */
	public function get_subscription_smart_tags() {

		return array(

			'subscription.id'                           => array(
				'description'       => __( 'Subscription ID', 'noptin-woocommerce' ),
				'callback'          => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'           => 'subscription.id',
				'conditional_logic' => 'number',
			),

			'subscription.date_created'                 => array(
				'description'       => __( 'Subscription creation date', 'noptin-woocommerce' ),
				'callback'          => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'           => 'subscription.date_created localize=1',
				'conditional_logic' => 'date',
			),

			'subscription.trial_end'                    => array(
				'description'       => __( 'Subscription trial end date', 'noptin-woocommerce' ),
				'callback'          => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'           => 'subscription.trial_end localize=1',
				'conditional_logic' => 'date',
			),

			'subscription.next_payment'                 => array(
				'description'       => __( 'Subscription next payment date', 'noptin-woocommerce' ),
				'callback'          => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'           => 'subscription.next_payment localize=1',
				'conditional_logic' => 'date',
			),

			'subscription.last_order_date_created'      => array(
				'description'       => __( 'Subscription last order creation date', 'noptin-woocommerce' ),
				'callback'          => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'           => 'subscription.last_order_date_created localize=1',
				'conditional_logic' => 'date',
			),

			'subscription.end'                          => array(
				'description'       => __( 'Subscription end date', 'noptin-woocommerce' ),
				'callback'          => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'           => 'subscription.end localize=1',
				'conditional_logic' => 'date',
			),

			'subscription.end_of_prepaid_term'          => array(
				'description'       => __( 'Subscription end of prepaid term date', 'noptin-woocommerce' ),
				'callback'          => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'           => 'subscription.end_of_prepaid_term localize=1',
				'conditional_logic' => 'date',
			),

			'subscription.date_paid'                    => array(
				'description'       => __( 'Last Payment Date', 'noptin-woocommerce' ),
				'callback'          => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'           => "subscription.date_paid format='date'",
				'conditional_logic' => 'date',
			),

			'subscription.date_completed'               => array(
				'description'       => __( 'Last order completed date', 'noptin-woocommerce' ),
				'callback'          => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'           => "subscription.date_completed format='date'",
				'conditional_logic' => 'date',
			),

			'subscription.formatted_subscription_total' => array(
				'description' => __( 'Formatted subscription total', 'noptin-woocommerce' ),
				'callback'    => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'     => 'subscription.formatted_subscription_total',
			),

			'subscription.view_subscription_url'        => array(
				'description' => __( 'View subscription URL', 'noptin-woocommerce' ),
				'callback'    => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'     => 'subscription.view_subscription_url',
			),

			'subscription.sign_up_fee'                  => array(
				'description'       => __( 'Subscription sign up fee', 'noptin-woocommerce' ),
				'callback'          => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'           => "subscription.sign_up_fee format='price'",
				'conditional_logic' => 'number',
			),

			'subscription.change_payment_method_url'    => array(
				'description' => __( 'Change payment method URL', 'noptin-woocommerce' ),
				'callback'    => '\Noptin\WooCommerce\Triggers\Subscription_Trigger::get_subscription_field',
				'example'     => 'subscription.change_payment_method_url',
			),
		);
	}

	/**
	 * Returns a subscription field.
	 *
	 * @param  array $args
	 * @return string
	 */
	public static function get_subscription_field( $args, $field = 'subscription.id' ) {

		if ( empty( self::$subscription ) ) {
			return '';
		}

		$subscription = self::$subscription;
		$field        = str_replace( 'subscription.', '', $field );

		switch ( $field ) {

			case 'id':
				$value = $subscription->get_id();
				break;

			case 'date_created':
			case 'trial_end':
			case 'next_payment':
			case 'last_order_date_created':
			case 'end':
			case 'end_of_prepaid_term':
				$timestamp = $subscription->get_time( $field );

				if ( empty( $timestamp ) ) {
					return '';
				}

				if ( empty( $args['localize'] ) ) {
					return gmdate( 'Y-m-d H:i:s', $timestamp );
				}

				return date_i18n( wc_date_format(), $timestamp );
			case 'date_paid':
			case 'date_completed':
				$value = 'date_paid' === $field ? $subscription->get_date_paid() : $subscription->get_date_completed();

				if ( empty( $value ) ) {
					return '';
				}

				if ( empty( $args['localize'] ) ) {
					return $value->date( 'Y-m-d H:i:s' );
				}

				return $value->date_i18n( wc_date_format() );

			case 'formatted_subscription_total':
				$value = $subscription->get_formatted_order_total();
				break;

			case 'view_subscription_url':
				$value = $subscription->get_view_order_url();
				break;

			case 'sign_up_fee':
				$value = $subscription->get_sign_up_fee();
				break;

			case 'change_payment_method_url':
				$value = $subscription->get_change_payment_method_url();
				break;

			default:
				$value = $subscription->get_meta( $field );
		}

		// Format the value.
		if ( ! empty( $args['format'] ) ) {

			switch ( $args['format'] ) {

				case 'price':
					$value = wc_price( $value );
					break;

			}
		}

		return $value;
	}

	/**
	 * Triggers a susbcription event.
	 *
	 * @param \WC_Subscription $subscription
	 */
	public function trigger_for_subscription( $subscription ) {

		// If we have a subscription ID, convert it to a subscription object.
		if ( is_numeric( $subscription ) ) {
			$subscription = wcs_get_subscription( $subscription );
		}

		// Bail if we don't have a subscription object.
		if ( ! is_a( $subscription, 'WC_Subscription' ) ) {
			return;
		}

		$customer = $this->get_subscription_customer( $subscription );

		// Prepare the trigger.
		$args = $this->before_trigger_wc( $subscription->get_last_order( 'all' ), $customer );

		self::$subscription = $subscription;

		$args['email']           = $subscription->get_billing_email();
		$args['subscription_id'] = $subscription->get_id();

		$this->trigger( $customer, $args );

		self::$subscription = false;
	}

	/**
	 * Prepares email test data.
	 *
	 * @since 1.11.0
	 * @param Noptin_Automation_Rule $rule
	 * @return Noptin_Automation_Rules_Smart_Tags
	 * @throws Exception
	 */
	public function get_test_smart_tags( $rule ) {

		// Fetch the last subscription.
		/** @var \WC_Subscription */
		$subscription = current( wcs_get_subscriptions( array( 'subscriptions_per_page' => 1 ) ) );

		if ( empty( $subscription ) ) {
			throw new \Exception( __( 'No subscriptions found', 'noptin-woocommerce' ) );
		}

		$customer = $this->get_subscription_customer( $subscription );

		// Prepare the trigger.
		$args = $this->before_trigger_wc(
			$subscription->get_last_order( 'all' ),
			$customer
		);

		$args = $this->prepare_trigger_args( $customer, $args );

		self::$subscription = $subscription;

		return $args['smart_tags'];
	}

	/**
	 * Serializes the trigger args.
	 *
	 * @since 1.11.1
	 * @param array $args The args.
	 * @return false|array
	 */
	public function serialize_trigger_args( $args ) {
		return array(
			'subscription_id' => $args['subscription_id'],
		);
	}

	/**
	 * Unserializes the trigger args.
	 *
	 * @since 1.11.1
	 * @param array $args The args.
	 * @return array|false
	 */
	public function unserialize_trigger_args( $previous_args ) {

		$subscription = wcs_get_subscription( $previous_args['subscription_id'] );

		// Bail if we don't have a subscription object.
		if ( ! is_a( $subscription, 'WC_Subscription' ) ) {
			throw new \Exception( 'The subscription no longer exists' );
		}

		$customer = $this->get_subscription_customer( $subscription );

		// Prepare the trigger.
		$args = array_merge(
			$previous_args,
			$this->before_trigger_wc(
				$subscription->get_last_order( 'all' ),
				$customer
			)
		);

		self::$subscription = $subscription;

		$args['email']           = $subscription->get_billing_email();
		$args['subscription_id'] = $subscription->get_id();

		// Prepare the trigger args.
		return $this->prepare_trigger_args( $customer, $args );
	}

	/**
	 * Retrieves subscription customer.
	 *
	 * @since 1.11.1
	 * @param \WC_Subscription $subscription The subscription.
	 * @return \WC_Customer
	 */
	public function get_subscription_customer( $subscription ) {
		$customer = new \WC_Customer( $subscription->get_customer_id() );

		// Set customer data from order if customer is not found.
		if ( ! $customer->get_id() ) {
			$customer->set_email( $subscription->get_billing_email() );
			$customer->set_billing_email( $subscription->get_billing_email() );
			$customer->set_first_name( $subscription->get_billing_first_name() );
			$customer->set_last_name( $subscription->get_billing_last_name() );
		}

		return $customer;
	}

}
