<?php

namespace Stratum_Menu\Managers;

use \Elementor\Plugin;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

/**
 * Class Post_Type_Manager
 * @package Stratum_Menu
 */
class Post_Type_Manager {
	protected $post_type = 'stratum-menu';
	protected $meta_key = 'stratum-menu-item';

	private $version;
	private $prefix;

	/**
	 * Post_Type_Manager constructor.
	 */
	public function __construct() {

		$settings = \Stratum_Menu\Settings::get_instance();

		$this->version = $settings->getVersion();
		$this->prefix  = $settings->getPrefix();

		$this->init();
	}

	public function init() {

		//==============Actions==============
		add_action( 'init', [ $this, 'register_post_type' ] );

		add_action( 'wp_ajax_stratum_menu_item', [ $this, 'stratum_menu_item' ] );
		add_action( 'wp_ajax_nopriv_stratum_menu_item', [ $this, 'stratum_menu_item' ] );

		add_action( 'wp_ajax_stratum-set-menu-item-svg', [ $this, 'wp_ajax_stratum_set_menu_item_svg' ] );

		//Add CPT Support to Elementor
		add_filter( 'option_elementor_cpt_support', [ $this, 'add_elementor_support' ] );
		add_filter( 'default_option_elementor_cpt_support', [ $this, 'add_elementor_support' ] );

		//Add template to menu items
		add_action( 'template_include', [ $this, 'menu_template' ], 9999 );

		add_action( 'admin_footer', [ $this, 'mega_menu_popup' ] );

		add_action( 'wp_nav_menu_item_custom_fields', [ $this, 'custom_menu_fields' ], 10, 5 ); //Add metabox to menu
		add_action( 'wp_update_nav_menu_item', [ $this, 'custom_menu_update' ], 10, 2 ); //Save extra menu fields
	}

	public function get_only_html( $item_id ) {
		$thumbnail_id = get_post_thumbnail_id( $item_id );

		$content = sprintf(
			'<label for="edit-menu-item-icon-svg-%1$s">%2$s</label><br /><a data-item-id="%1$s" class="stratum-set-post-thumbnail button">%3$s</a>%4$s<input class="edit-menu-img" name="menu-item-icon-svg[%1$s]" type="hidden" value="%1$s" />',
			$item_id,
			esc_html__( 'Icon SVG', 'stratum-menu' ),
			$thumbnail_id ? wp_get_attachment_image( $thumbnail_id ) : esc_html__( 'Select Image', 'stratum-menu' ),
			$thumbnail_id ? '<a href="#" class="stratum-remove-post-thumbnail">' . esc_html__( 'Remove', 'stratum-menu' ) . '</a>' : ''
		);

		return $content;
	}

	/**
	 * Update item thumbnail via ajax action.
	 *
	 * @since 2.0
	 */
	public function wp_ajax_stratum_set_menu_item_svg() {
		$json = ! empty( $_REQUEST['json'] );
		$post_ID = intval( $_POST['post_id'] );

		if ( ! current_user_can( 'edit_post', $post_ID ) ) {
			wp_die( - 1 );
		}

		$thumbnail_id = intval( $_POST['thumbnail_id'] );

		check_ajax_referer( 'update-menu-item' );

		if ( $thumbnail_id == '-1' ) {
			$success = delete_post_thumbnail( $post_ID );
		} else {
			$success = set_post_thumbnail( $post_ID, $thumbnail_id );
		}

		if ( $success ) {
			$return = $this->get_only_html( $post_ID );
			$json ? wp_send_json_success( $return ) : wp_die( $return );
		}

		wp_die( 0 );
	}

	/**
	 * When menu item removed remove menu image metadata.
	 */
	public function menu_image_delete_menu_item_image_action() {

		$menu_item_id = (int) $_REQUEST['menu-item'];

		check_admin_referer( 'delete-menu_item_image_' . $menu_item_id );

		if ( is_nav_menu_item( $menu_item_id ) && has_post_thumbnail( $menu_item_id ) ) {
			delete_post_thumbnail( $menu_item_id );
		}
	}

	public function mega_menu_popup() {
		$screen = get_current_screen();
		if (get_current_screen()->id == 'nav-menus'){
			echo "
				<div id='stratum-menu-wrapper'>
					<div class='stratum-menu-popup'>
						<div class='stratum-menu-overlay'></div>
						<div class='stratum-menu-popup-loader'><span class='loader-popup-icon dashicons dashicons-wordpress-alt'></span></div>
						<div class='stratum-menu-container'>
							<div class='stratum-menu-popup-close'><span class='close-popup-icon dashicons dashicons-no-alt'></span></div>
							<div class='stratum-menu-content'></div>
						</div>
					</div>
				</div>
			";
		}
	}

	public function custom_menu_fields( $item_id, $item, $depth, $args, $id ) {
		extract( shortcode_atts( array(
			'menu_item_color' => '',
			'menu_icon_type' => 'icon_type',
			'menu_item_icon' => '',
			'menu_item_icon_svg' => '',
			'menu_icon_color' => '',
			'menu_badge_text' => '',
			'menu_badge_color' => '',
			'menu_badge_background' => '',
			'menu_enable_mega_menu' => 'no',
			'mega_menu_id' => null,
		), get_post_meta( $item_id, $this->meta_key(), true )) );

	?>

		<p class="description description-wide">
			<label><?php esc_html_e( 'Item Color', 'stratum-menu' ); ?><br /></label>
			<input class="stratum-menu-color" name="menu-item-color[<?php echo esc_attr($item_id);?>]" value="<?php echo esc_attr($menu_item_color);?>" type="text">
		</p>

		<p class="description description-wide">
			<label for="edit-menu-icon-type-<?php echo esc_attr($item_id);?>">
				<?php esc_html_e( 'Icon Type', 'stratum-menu' ); ?><br />
				<select class="widefat stratum-menu-select-type" id="edit-menu-icon-type-<?php echo esc_attr($item_id);?>" name="menu-icon-type[<?php echo esc_attr($item_id);?>]">
					<option value="icon_type"<?php echo ( $menu_icon_type == 'icon_type' ? ' selected' : '' ); ?>>
						<?php esc_html_e( 'Icon Library', 'stratum-menu' ); ?>
					</option>
					<option value="icon_svg"<?php echo ( $menu_icon_type == 'icon_svg' ? ' selected' : '' ); ?>>
						<?php esc_html_e( 'Custom SVG', 'stratum-menu' ); ?>
					</option>
				</select>
			</label>
		</p>

		<p id="stratum-lib-icon" class="description description-thin">
			<label for="edit-menu-item-icon-<?php echo esc_attr($item_id);?>">
				<?php esc_html_e( 'Item Icon', 'stratum-menu' ); ?><br />
				<select class="widefat stratum-menu-select" id="edit-menu-item-icon-<?php echo esc_attr($item_id);?>" name="menu-item-icon[<?php echo esc_attr($item_id);?>]">
					<option></option>
					<?php
						$icons = \Elementor\Control_Icon::get_icons();
						foreach ($icons as $key => $value) {
							echo "<option value='".esc_attr($key)."'".($key == $menu_item_icon ? ' selected' : '').">".esc_attr($value)."</option>";
						}
					?>
				</select>
			</label>
		</p>

		<p id="stratum-custom-icon" class="description description-thin icon-item-id-<?php echo $item_id; ?>">
			<label for="edit-menu-item-icon-svg-<?php echo esc_attr($item_id);?>">
				<?php esc_html_e( 'Icon SVG', 'stratum-menu' ) ?>
			</label><br />
			<?php
				$thumbnail_id = get_post_thumbnail_id( $item_id );

				printf(
					'<a data-item-id="%s" class="stratum-set-post-thumbnail button">%s</a>%s',
					$item_id,
					$thumbnail_id ? wp_get_attachment_image( $thumbnail_id ) : esc_html__( 'Select Image', 'stratum-menu' ),
					$thumbnail_id ? '<a href="#" class="stratum-remove-post-thumbnail">' . esc_html__( 'Remove', 'stratum-menu' ) . '</a>' : ''
				);
			?>

			<input class="edit-menu-img" name="menu-item-icon-svg[<?php echo esc_attr($item_id);?>]" type="hidden" value="<?php echo esc_attr($item_id); ?>" />
		</p>

		<p class="description description-thin">
			<label><?php esc_html_e( 'Icon Color', 'stratum-menu' ); ?><br /></label>
			<input class="stratum-menu-color" name="menu-icon-color[<?php echo esc_attr($item_id);?>]" value="<?php echo esc_attr($menu_icon_color);?>" type="text">
		</p>

		<p class="description description-wide">
			<label for="edit-menu-item-attr-badge-text-<?php echo esc_attr($item_id);?>">
				<?php esc_html_e( 'Badge Text', 'stratum-menu' ); ?><br />
				<input class="widefat edit-menu-item-attr-title" name="menu-badge-text[<?php echo esc_attr($item_id);?>]" value="<?php echo esc_attr($menu_badge_text);?>" type="text" id="edit-menu-item-attr-badge-text-<?php echo esc_attr($item_id);?>"/>
			</label>
		</p>

		<p class="description description-thin">
			<label><?php esc_html_e( 'Badge Color', 'stratum-menu' ); ?><br /></label>
			<input class="stratum-menu-color" name="menu-badge-color[<?php echo esc_attr($item_id);?>]" value="<?php echo esc_attr($menu_badge_color);?>" type="text">
		</p>

		<p class="description description-thin">
			<label><?php esc_html_e( 'Badge Background', 'stratum-menu' ); ?><br /></label>
			<input class="stratum-menu-color" name="menu-badge-background[<?php echo esc_attr($item_id);?>]" value="<?php echo esc_attr($menu_badge_background);?>" type="text">
		</p>
		<div class="wp-clearfix"></div>
		<div class="wp-clearfix" style="margin-top:15px;margin-bottom:15px;">
			<p class="description description-wide">
				<label>
					<input type="checkbox" <?php echo checked( 'yes', $menu_enable_mega_menu, false ); ?> name="menu-enable-mega-menu[<?php echo esc_attr($item_id);?>]">
					<?php esc_html_e( 'Enable Mega Menu', 'stratum-menu' ); ?>
				</label>
			</p>

			<p class="description description-wide">
				<input type="button" class="button stratum-menu-button" value="<?php
					esc_html_e( 'Edit Mega Menu', 'stratum-menu' ); ?>" data-item_id="<?php echo esc_attr($item_id);?>">
			</p>

			<?php

			$custom_menu_obj = get_post_meta( $item_id, $this->meta_key(), true );

			if ( $custom_menu_obj && is_array( $custom_menu_obj ) ) { ?>

			<p class="description description-wide">
			<?php
				$menu_post_id = $custom_menu_obj['mega_menu_id'];
				if ( $menu_post_id && get_post_status( $menu_post_id ) ) {
					printf(
						__('Mega Menu Element ID: %1$s', 'stratum-menu'),
						sprintf(
							'%s%s%s',
							'<a target="_blank" href="' . get_edit_post_link( $menu_post_id ) . '">',
							$menu_post_id,
							'</a>'
						)
					);
				}
			?>
			</p>
		</div>
		<?php }
	}

	function custom_menu_update( $menu_id, $menu_item_id ) {
		extract( shortcode_atts( array(
			'mega_menu_id' => null,
		), get_post_meta( $menu_item_id, $this->meta_key(), true )) );

		$custom_menu_obj = array(
			'menu_item_color' => isset( $_POST[ 'menu-item-color' ][ $menu_item_id ] ) ? $_POST[ 'menu-item-color' ][ $menu_item_id ] : '',
			'menu_icon_type' => isset( $_POST[ 'menu-icon-type' ][ $menu_item_id ] ) ? $_POST[ 'menu-icon-type' ][ $menu_item_id ] : 'icon_type',
			'menu_item_icon' => isset( $_POST[ 'menu-item-icon' ][ $menu_item_id ] ) ? $_POST[ 'menu-item-icon' ][ $menu_item_id ] : '',
			'menu_item_icon_svg' => isset( $_POST[ 'menu-item-icon-svg' ][ $menu_item_id ] ) ? $_POST[ 'menu-item-icon-svg' ][ $menu_item_id ] : '',
			'menu_icon_color' => isset( $_POST[ 'menu-icon-color' ][ $menu_item_id ] ) ? $_POST[ 'menu-icon-color' ][ $menu_item_id ] : '',
			'menu_badge_text' => isset( $_POST[ 'menu-badge-text' ][ $menu_item_id ] ) ? $_POST[ 'menu-badge-text' ][ $menu_item_id ] : '',
			'menu_badge_color' => isset( $_POST[ 'menu-badge-color' ][ $menu_item_id ] ) ? $_POST[ 'menu-badge-color' ][ $menu_item_id ] : '',
			'menu_badge_background' => isset( $_POST[ 'menu-badge-background' ][ $menu_item_id ] ) ? $_POST[ 'menu-badge-background' ][ $menu_item_id ] : '',
			'menu_enable_mega_menu' => (isset( $_POST[ 'menu-enable-mega-menu' ][ $menu_item_id ] ) && $_POST[ 'menu-enable-mega-menu' ][ $menu_item_id ] == 'on') ? 'yes' : 'no',
			'mega_menu_id' => $mega_menu_id,
		);

		update_post_meta( $menu_item_id, $this->meta_key(), $custom_menu_obj );
	}

	public function slug() {
		return $this->post_type;
	}

	public function meta_key() {
		return $this->meta_key;
	}

	public function menu_template( $template ) {

		if ( is_singular( $this->slug() ) ) {
			$template = stratum_menu_get_plugin_path( '/includes/editor-template/single.php' );
			if ( Plugin::instance()->preview->is_preview_mode() ) {
				$template = stratum_menu_get_plugin_path( '/includes/editor-template/preview.php' );
			}
			return $template;
		}

		return $template;
	}

	public function register_post_type() {

		$labels = array(
			'name'          => esc_html__( 'Mega Menu Elements', 'stratum-menu' ),
			'singular_name' => esc_html__( 'Mega Menu Element', 'stratum-menu' ),
			'menu_name'     => esc_html__( 'Mega Menu Elements', 'stratum-menu' ),
		);

		$supports = apply_filters( 'stratum-menu/post-type/register/supports', [ 'title', 'author' ] );

		/*
		 * https://developer.wordpress.org/reference/functions/register_post_type/
		 */

		$args = array(
			'labels'              => $labels,
			'taxonomies'          => [],
			'public'              => false,
			'show_ui'             => true,
			'show_in_menu'        => 'themes.php', //[true | false | themes.php]
			'show_in_admin_bar'   => false,
			'publicly_queryable'  => true,
			'can_export'          => true,
			'rewrite'             => false,
			'capability_type'     => 'post',
			'supports'            => $supports,
			'menu_position'       => 5,
			'map_meta_cap'			 => true,
			'capabilities'			 => array(
				'create_posts' => false,
			),
		);

		register_post_type( 'stratum-menu', $args );
	}

	public function stratum_menu_item() {

		if ( ! current_user_can( 'edit_theme_options' ) ) {
			return;
		}

		$menu_id = intval( $_GET['menu_id'] );
		$item_id = intval( $_GET['item_id'] );

		extract( shortcode_atts( array(
			'mega_menu_id' => null,
		), get_post_meta( $item_id, $this->meta_key(), true )) );

		//Fix delete post
		if (!is_null(get_post($mega_menu_id))){
			$current_post = get_post($mega_menu_id);
			if ($current_post->post_status == 'trash'){
				wp_delete_post($mega_menu_id, true);
			}
		}

		if ( !$mega_menu_id || is_null(get_post($mega_menu_id)) ) {

			$mega_menu_id = wp_insert_post( array(
				'post_title'  => 'stratum-menu-item-' . $item_id,
				'post_status' => 'publish',
				'comment_status' => 'closed',
				'ping_status' => 'closed',
				'post_type'   => $this->slug(),
				'post_content' => ''
			) );

			$menu_obj = array(
				'menu_item_color' => '',
				'menu_icon_type' => 'icon_type',
				'menu_item_icon' => '',
				'menu_item_icon_svg' => '',
				'menu_icon_color' => '',
				'menu_badge_text' => '',
				'menu_badge_color' => '',
				'menu_badge_background' => '',
				'menu_enable_mega_menu' => 'no',
				'mega_menu_id' => $mega_menu_id
			);

			update_post_meta( $item_id, $this->meta_key(), $menu_obj );
		}

		$post_link = add_query_arg(
			array(
				'post'        => $mega_menu_id,
				'action'      => 'elementor',
				'context'     => 'stratum-menu',
				'parent_menu' => $menu_id,
			),
			admin_url( 'post.php' )
		);

		wp_redirect( $post_link );
		die();
	}

	public function add_elementor_support( $value ) {

		if ( empty( $value ) ) {
			$value = array();
		}

		return array_merge( $value, array( $this->slug() ) );
	}
}
