<?php
/**
 * Noptin Sync.
 *
 */

defined( 'ABSPATH' ) || exit;

/**
 * Noptin Sync.
 *
 */
class Noptin_Sync {

	/**
	 * Class Constructor.
	 */
	public function __construct() {

		add_filter( 'noptin_get_setting_sections', array( $this, 'filter_setting_tabs' ) );
		add_filter( 'noptin_get_settings', array( $this, 'filter_settings' ) );
		add_filter( 'noptin_sanitize_settings', array( $this, 'sanitize_settings' ) );
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );

	}

	/**
	 * Filters setting tabs.
	 *
	 * @param array $tabs
	 */
	public function filter_setting_tabs( $tabs ) {
		$tabs['sync'] = __( 'Subscriber Sync', 'noptin-addons-pack' );
		return $tabs;
	}

	/**
	 * Filters noptin settings.
	 *
	 * @param array $settings
	 */
	public function filter_settings( $settings ) {

		$settings['sync_url'] = array(
			'el'          => 'input',
			'type'        => 'text',
			'section'     => 'sync',
			'label'       => __( 'Sync URL', 'noptin-addons-pack' ),
			'description' => __( 'Use this URL when syncing subscribers from another site.', 'noptin-addons-pack' ),
			'readonly'    => 'readonly',
			'onclick'     => 'this.select()',
			'default'     => rest_url( 'noptin/1/sync' ),
		);

		$settings['sync_secret'] = array(
			'el'          => 'input',
			'type'        => 'text',
			'section'     => 'sync',
			'label'       => __( 'Sync Secret', 'noptin-addons-pack' ),
			'description' => __( 'Use this Secret when syncing subscribers from another site.', 'noptin-addons-pack' ),
			'default'     => $this->get_default_sync_secret(),
		);

		return $settings;
	}

	/**
	 * Retrieves the sync secret.
	 *
	 */
	public function get_sync_secret() {
		return get_noptin_option( 'sync_secret', $this->get_default_sync_secret() );
	}

	/**
	 * Retrieves the default sync secret.
	 *
	 */
	public function get_default_sync_secret() {
		$default = get_option( 'default_noptin_sync_secret' );

		if ( empty( $default ) ) {
			$default = wp_generate_password( 32, true );
			update_option( 'default_noptin_sync_secret', $default );
		}

		return $default;
	}

	/**
	 * Sanitizes settings.
	 *
	 * @param array $settings
	 */
	public function sanitize_settings( $settings ) {

		if ( isset( $settings['sync_url'] ) ) {
			unset( $settings['sync_url'] );
		}

		return $settings;
	}

	/**
	 * Registers routes for each namespace.
	 *
	 * @since 1.0.0
	 *
	 */
	public function register_routes() {

		// List all registered tabs.
		register_rest_route(
			'noptin/1',
			'sync',
			array(
				array(
					'methods'             => WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'sync_subscriber' ),
					'permission_callback' => array( $this, 'can_sync_subscriber' ),
				),
				'schema' => '__return_empty_array',
			)
		);

	}

	/**
	 * Verifies the secret key.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|boolean
	 */
	public function can_sync_subscriber( $request ) {

		if ( $request->get_param( 'secret' ) !== $this->get_sync_secret() ) {
			return new WP_Error( 'unauthorized', __( 'Sorry, you do not have permission to do this.', 'noptin-addons-pack' ), array( 'status' => 403 ) );
		}

		return true;
	}

	/**
	 * Syncs the subscriber.
	 *
	 * @since 1.0.0
	 * @param WP_REST_Request $request Full data about the request.
	 * @return WP_Error|WP_REST_Response
	 */
	public function sync_subscriber( $request ) {

		$subscriber_data = $request->get_params();

		if ( empty( $subscriber_data['email'] ) ) {
			return rest_ensure_response( true );
		}

		$subscriber_id   = get_noptin_subscriber_id_by_email( $subscriber_data['email'] );
		$subscriber_meta = empty( $subscriber_data['meta'] ) ? array() : $subscriber_data['meta'];
		$subscriber_args = empty( $subscriber_data['args'] ) ? array() : $subscriber_data['args'];

		unset( $subscriber_data['secret'], $subscriber_data['meta'], $subscriber_data['args'], $subscriber_data['confirm_key'], $subscriber_data['id'] );

		foreach ( $subscriber_meta as $meta_key => $values ) {
			$subscriber_data[ $meta_key ] = $values[0];
		}

		if ( empty( $subscriber_id ) ) {
			$subscriber_id = add_noptin_subscriber( $subscriber_data );
		} else {
			update_noptin_subscriber( $subscriber_id, $subscriber_args );
		}

		foreach ( $subscriber_meta as $meta_key => $values ) {
			foreach ( $values as $value ) {
				if ( get_noptin_subscriber_meta( $subscriber_id, $meta_key, true ) !== $value ) {
					add_noptin_subscriber_meta( $subscriber_id, $meta_key, $value );
				}
			}
		}

		do_action( 'noptin_synced_subscrier', $subscriber_id, $subscriber_args );
		return rest_ensure_response( true );
	}

}
new Noptin_Sync();
