<?php

namespace Noptin\Addons_Pack\WooCommerce;

/**
 * Bulk Emails API: Email Sender (customers).
 *
 * Contains the main email sender class.
 *
 * @since   1.12.0
 * @package Noptin
 */

defined( 'ABSPATH' ) || exit;

/**
 * The mass mailer class for sending emails to customers.
 */
class Email_Sender_Customers extends \Hizzle\Noptin\Bulk_Emails\Email_Sender {

	/**
	 * The email sender.
	 * @var string
	 */
	protected $sender = 'woocommerce_customers';

	/**
	 * Displays newsletter sending options.
	 *
	 * @param \Noptin_Newsletter_Email|\Noptin_automated_Email $campaign
	 *
	 * @return bool
	 */
	public function display_sending_options( $campaign ) {

		// Render sender options.
		$options = $campaign->get( 'woocommerce_sender_options' );
		$options = is_array( $options ) ? $options : array();

		$fields  = array(
			'paying'            => array(
				'label' => __( 'Only send to paying customers', 'noptin-addons-pack' ),
				'type'  => 'checkbox',
			),

			'billing'           => array(
				'label'       => __( 'Billing Address', 'noptin-addons-pack' ),
				'type'        => 'region',
				'description' => __( 'Optional. Limit recipients to customers in a given billing region.', 'noptin-addons-pack' ),
			),

			'shipping'          => array(
				'label'       => __( 'Shipping Address', 'noptin-addons-pack' ),
				'type'        => 'region',
				'description' => __( 'Optional. Limit recipients to customers in a given shipping region.', 'noptin-addons-pack' ),
			),

			'bought'            => array(
				'label'       => __( 'Bought', 'noptin-addons-pack' ),
				'type'        => 'text',
				'placeholder' => sprintf(
					// translators: %s: Example product ids.
					__( 'For example, %s', 'noptin-addons-pack' ),
					'12, 13'
				),
				'description' => __( 'Optional. Provide a comma separated list of product ids if you only want to send this email to customers who have BOUGHT specific products', 'noptin-addons-pack' ),
			),

			'bought_not'        => array(
				'label'       => __( "Haven't Bought", 'noptin-addons-pack' ),
				'type'        => 'text',
				'placeholder' => sprintf(
					// translators: %s: Example product ids.
					__( 'For example, %s', 'noptin-addons-pack' ),
					'10, 30'
				),
				'description' => __( 'Optional. Provide a comma separated list of product ids if you only want to send this email to customers who have NOT BOUGHT specific products', 'noptin-addons-pack' ),
			),

			'last_order_before' => array(
				'label'       => __( 'Last ordered before', 'noptin-addons-pack' ),
				'type'        => 'text',
				// translators: %s: Example date.
				'placeholder' => sprintf( __( 'For example, %s', 'noptin-addons-pack' ), gmdate( 'Y-m-d' ) ),
				'description' => __( 'Optional. Enter a date if you only want to send this email to customers whose last order was BEFORE the date', 'noptin-addons-pack' ),
			),

			'last_order_after'  => array(
				'label'       => __( 'Last ordered after', 'noptin-addons-pack' ),
				'type'        => 'text',
				// translators: %s: Example date.
				'placeholder' => sprintf( __( 'For example, %s', 'noptin-addons-pack' ), gmdate( 'Y-m-d', time() - 2 * DAY_IN_SECONDS ) ),
				'description' => __( 'Optional. Enter a date if you only want to send this email to customers whose last order was AFTER the date', 'noptin-addons-pack' ),
			),

		);

		// Multi-lingual support.
		if ( noptin_is_multilingual() ) {
			$fields['user_locale'] = array(
				'label'   => __( 'Preferred Language', 'noptin-addons-pack' ),
				'type'    => 'select',
				'options' => array_merge(
					array(
						'' => __( 'Any', 'noptin-addons-pack' ),
					),
					noptin_get_available_languages()
				),
			);
		}

		$fields = apply_filters( 'noptin_woocommerce_sending_options', $fields, $campaign );

		$this->display_sending_fields( $campaign, 'woocommerce_sender_options', $fields );
	}

	/**
	 * Displays a region select field.
	 *
	 * @param array $field
	 *
	 * @return bool
	 */
	public function display_sending_field_region( $field ) {

		$countries = WC()->countries->get_countries();

		?>
			<p>
				<label>
					<strong><?php echo wp_kses_post( $field['label'] ); ?></strong>
					<select name="<?php echo esc_attr( $field['value'] ); ?>" class="noptin-select2 widefat" style="width: 100%;">
						<option value="" <?php selected( empty( $value ) ); ?>><?php esc_html_e( 'Any', 'noptin-addons-pack' ); ?></option>
						<?php
							foreach ( WC()->countries->get_continents() as $continent_code => $continent ) {
								echo '<option value="continent:' . esc_attr( $continent_code ) . '"' . selected( "continent:$continent_code", $field['value'], false ) . '>' . esc_html( $continent['name'] ) . '</option>';

								foreach ( $continent['countries'] as $country_code ) {
									echo '<option value="country:' . esc_attr( $country_code ) . '"' . selected( "country:$country_code", $field['value'], false ) . '>' . esc_html( '&nbsp;&nbsp; ' . $countries[ $country_code ] ) . '</option>';

									$states = WC()->countries->get_states( $country_code );

									if ( $states ) {
										foreach ( $states as $state_code => $state_name ) {
											echo '<option value="state:' . esc_attr( $country_code . ':' . $state_code ) . '"' . selected( "state:$country_code:$state_code", $field['value'], false ) . '>' . esc_html( '&nbsp;&nbsp;&nbsp;&nbsp; ' . $state_name . ', ' . $countries[ $country_code ] ) . '</option>';
										}
									}
								}
							}
						?>
					</select>
				</label>
				<?php echo wp_kses_post( $field['description'] ); ?>
			</p>
		<?php

	}

	/**
	 * Sends a single email to a subscriber.
	 *
	 * @param \Noptin_Newsletter_Email $campaign
	 * @param int $recipient
	 *
	 * @return bool
	 */
	public function send( $campaign, $recipient ) {

		$customer = new \WC_Customer( $recipient );
		$key      = '_campaign_' . $campaign->id;
		$email    = $customer->get_email();

		// Bail if the customer is not found or is unsubscribed...
		if ( empty( $email ) || noptin_is_email_unsubscribed( $email ) ) {
			return null;
		}

		// ... or does not qualify for the campaign.
		if ( ! $this->can_email_customer( $campaign, $customer ) ) {
			update_user_meta( $customer->get_id(), $key, 0 );
			return null;
		}

		// Generate and send the actual email.
		/**@var Noptin_WooCommerce_Lifetime_Value_Email */
		$helper = noptin()->emails->automated_email_types->types['woocommerce_lifetime_value'];
		$helper->prepare_customer( $customer );

		$result = noptin()->emails->newsletter->send( $campaign, $campaign->id, $customer->get_email() );

		$helper->clean_customer( $customer );

		// Log the send.
		update_user_meta( $customer->get_id(), $key, (int) $result );

		return $result;
	}

	/**
	 * Checks if a customer is valid for a given task.
	 *
	 * @param \Noptin_Newsletter_Email $campaign The current campaign.
	 * @param \WC_Customer $customer The customer to check.
	 * @return bool
	 */
	public function can_email_customer( $campaign, $customer ) {

		// Do not send twice.
		if ( '' !== get_user_meta( $customer->get_id(), '_campaign_' . $campaign->id, true ) ) {
			return false;
		}

		// Prepare sender options.
		$options = $campaign->get( 'woocommerce_sender_options' );
		$options = is_array( $options ) ? $options : array();

		// Get user locale.
		if ( ! empty( $options['locale'] ) && get_user_locale( $customer->get_id() ) !== $options['locale'] ) {
			return false;
		}

		// Check if the customer has bought required products.
		if ( ! empty( $options['bought'] ) && ! $this->has_bought_product( $customer->get_id(), $options['bought'] ) ) {
			return false;
		}

		// Ensure the customer has not bought a prohibited product.
		if ( ! empty( $options['bought_not'] ) && $this->has_bought_product( $customer->get_id(), $options['bought_not'] ) ) {
			return false;
		}

		// Check last order before and after.
		if ( ! empty( $options['last_order_before'] ) || ! empty( $options['last_order_after'] ) ) {

			$last_order = $customer->get_last_order();

			if ( empty( $last_order ) ) {
				return false;
			}

			if ( ! empty( $options['last_order_before'] ) && $last_order->get_date_created()->getTimestamp() > strtotime( $options['last_order_before'] ) ) {
				return false;
			}

			if ( ! empty( $options['last_order_after'] ) && $last_order->get_date_created()->getTimestamp() < strtotime( $options['last_order_after'] ) ) {
				return false;
			}
		}

		// Check whether or not is a paying customer.
		if ( ! empty( $options['paying'] ) && ! $customer->get_is_paying_customer() ) {
			return false;
		}

		// Check billing region.
		if ( ! empty( $options['billing'] ) && ! $this->is_in_region( $customer, $options['billing'], 'billing' ) ) {
			return false;
		}

		// Check shipping region.
		if ( ! empty( $options['shipping'] ) && ! $this->is_in_region( $customer, $options['shipping'], 'shipping' ) ) {
			return false;
		}

		// TODO: get_shipping_postcode, get_shipping_city, get_billing_postcode, get_billing_city
		// Customer, get_date_created, get_total_spent, get_order_count,
		// is_customer_outside_base
		return apply_filters( 'noptin_woocommerce_can_email_customer_for_campaign', true, $options, $customer, $campaign );
	}

	/**
	 * Checks if a customer has bought any of the given products.
	 *
	 * @param id $customer_id.
	 * @param array $product_ids
	 * @return bool
	 */
	public function has_bought_product( $customer_id, $product_ids ) {

		foreach ( wp_parse_id_list( $product_ids ) as $product_id ) {
			if ( wc_customer_bought_product( '', $customer_id, $product_id ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Checks if a customer is in a given region.
	 *
	 * @param WC_Customer $customer
	 * @param string $region
	 * @return bool
	 */
	public function is_in_region( $customer, $region, $mode = 'billing' ) {

		if ( empty( $region ) ) {
			return true;
		}

		// Set country and state.
		if ( 'billing' === $mode ) {
			$country = $customer->get_billing_country();
			$state   = $customer->get_billing_state();
		} else {
			$country = $customer->get_shipping_country();
			$state   = $customer->get_shipping_state();
		}

		// Region has the format type:code.
		$location_parts = explode( ':', $region );
		switch ( $location_parts[0] ) {

			case 'state':
				$customer_state   = strtoupper( wc_clean( $state ) );
				$customer_country = strtoupper( wc_clean( $country ) );
				$required_state   = strtoupper( wc_clean( $location_parts[2] ) );
				$required_country = strtoupper( wc_clean( $location_parts[1] ) );
				return $customer_state === $required_state && $customer_country === $required_country;

			case 'country':
				return strtoupper( wc_clean( $location_parts[1] ) ) === strtoupper( wc_clean( $country ) );

			case 'continent':
				return strtoupper( wc_clean( $location_parts[1] ) ) === strtoupper( wc_clean( WC()->countries->get_continent_code_for_country( $country ) ) );

		}

		return false;
	}

	/**
	 * Fired after a campaign is done sending.
	 *
	 * @param @param Noptin_Newsletter_Email $campaign
	 *
	 */
	public function done_sending( $campaign ) {
		global $wpdb;

		$wpdb->delete(
			$wpdb->usermeta,
			array(
				'meta_key' => '_campaign_' . $campaign->id,
			)
		);

	}

	/**
	 * Fetches relevant customers for the campaign.
	 *
	 * @param Noptin_Newsletter_Email $campaign
	 */
	public function get_recipients( $campaign ) {
		global $wpdb;

		// Get customer IDs.
		$customer_ids = $wpdb->get_col( "SELECT DISTINCT meta_value FROM {$wpdb->postmeta} WHERE meta_key = '_customer_user' AND meta_value > 0" );
		$customer_ids = wp_parse_id_list( $customer_ids );

		// Get all user ids whove received the campaign.
		$user_ids = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT DISTINCT user_id FROM {$wpdb->usermeta} WHERE meta_key = %s",
				'_campaign_' . $campaign->id
			)
		);
		$user_ids = wp_parse_id_list( $user_ids );

		// Remove users whove already received the campaign.
		return array_diff( $customer_ids, $user_ids );
	}

}
