<?php

namespace Noptin\Addons_Pack\Emails;

/**
 * Migrates from old emails to new emails.
 *
 * @since   2.0.0
 * @package Noptin
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Migrates from old emails to new emails.
 *
 * @since 2.0.0
 */
class Migrate {

	/**
	 * Migrations.
	 */
	protected $migrations = array(
		'welcome_email'   => 'automation_rule_new_subscriber',
		'welcome_user'    => 'automation_rule_new_user',
		'subscriber_list' => 'automation_rule_add_to_lists',
		'subscriber_tag'  => 'automation_rule_add_to_tags',
	);

	/**
	 * Constructor.
	 */
	public function __construct() {
		add_action( 'init', array( $this, 'maybe_migrate' ) );
	}

	/**
	 * Migrates old emails to new emails.
	 */
	public function maybe_migrate() {
		global $noptin_addons_pack_migrate_emails;

		// Abort if already migrated.
		if ( get_option( 'noptin_addons_pack_migrated_emails', false ) ) {
			return;
		}

		// Fetch all emails.
		$noptin_addons_pack_migrate_emails = get_posts(
			array(
				'post_type'   => 'noptin-campaign',
				'post_status' => array( 'pending', 'draft', 'future', 'publish' ),
				'numberposts' => -1,
			)
		);

		// Migrate lists.
		$this->migrate_lists();

		// Loop through all emails.
		foreach ( $noptin_addons_pack_migrate_emails as $email ) {

			// Abort if not an automated email.
			if ( 'automation' !== get_post_meta( $email->ID, 'campaign_type', true ) ) {
				continue;
			}

			$email_type = get_post_meta( $email->ID, 'automation_type', true );

			// Check if we're migrating this email type.
			if ( isset( $this->migrations[ $email_type ] ) ) {
				$email           = new \Noptin_Automated_Email( $email->ID );
				$email->type     = $this->migrations[ $email_type ];
				$automation_rule = noptin_get_automation_rule();

				if ( ! is_wp_error( $automation_rule ) ) {
					$automation_rule->set_trigger_id( $email->get_trigger() );
					$automation_rule->set_action_id( 'email' );

					$automation_rule->set_action_settings(
						array(
							'automated_email_id' => $email->id,
						)
					);

					$conditional_logic = call_user_func( array( $this, $email_type ), $email );

					if ( isset( $conditional_logic['custom_type'] ) ) {
						$email->type = $conditional_logic['custom_type'];

						$automation_rule->set_trigger_id( $email->get_trigger() );
						unset( $conditional_logic['custom_type'] );
					}

					$automation_rule->set_trigger_settings(
						array(
							'conditional_logic' => $conditional_logic,
						)
					);

					$automation_rule->save();

					if ( $automation_rule->exists() ) {
						$email->options['automation_rule'] = $automation_rule->get_id();
					}
				}

				// If recipients contains, [[, replace with [[email]]
				if ( strpos( $email->get_recipients(), '[[' ) !== false ) {
					$email->options['recipients'] = '[[email]]';
				}

				$email->save();
			}
		}

		// Migrate form emails.
		$this->migrate_form_welcome_emails();

		// Mark as migrated.
		update_option( 'noptin_addons_pack_migrated_emails', true );

		// Flush cache.
		wp_cache_flush();
	}

	/**
	 * Fetches subscriber conditional logic.
	 *
	 * @param \Noptin_Automated_Email $email
	 */
	protected function welcome_email( $email ) {
		$rules = array();

		// Handle custom fields.
		foreach ( array_keys( get_noptin_subscriber_filters() ) as $key ) {

			$value = $email->get( 'noptin_custom_field_' . $key );

			// Try retrieving the value directly.
			if ( '' === $value ) {
				$value = $email->get( $key );
			}

			// Filter.
			if ( '' !== $value ) {
				$rules[] = array(
					'condition' => 'is',
					'type'      => $key,
					'value'     => $value,
				);
			}
		}

		// Backwards compatibility for _subscriber_via.
		$source = $email->get( '_subscriber_via' );

		if ( '' !== $source ) {
			$rules[] = array(
				'condition' => 'is',
				'type'      => 'source',
				'value'     => $source,
			);
		}

		return array(
			'enabled' => ! empty( $rules ),
			'action'  => 'allow',
			'type'    => 'all',
			'rules'   => $rules,
		);

	}

	/**
	 * Fetches user conditional logic.
	 *
	 * @param \Noptin_Automated_Email $email
	 */
	protected function welcome_user( $email ) {
		$rules   = array();
		$options = $email->get( 'wp_users_options' );
		$roles   = isset( $options['roles'] ) ? $options['roles'] : array();

		// Handle custom fields.
		foreach ( array_filter( wp_parse_list( $roles ) ) as $role ) {
			$rules[] = array(
				'condition' => 'is_not',
				'type'      => 'user_role',
				'value'     => $role,
			);
		}

		return array(
			'enabled' => ! empty( $rules ),
			'action'  => 'allow',
			'type'    => 'all',
			'rules'   => $rules,
		);

	}

	/**
	 * Fetches list conditional logic.
	 *
	 * @param \Noptin_Automated_Email $email
	 */
	protected function subscriber_list( $email ) {

		$list   = $email->get( 'subscriber_list' );
		$action = $email->get( 'list_action' );
		$rules  = array();

		if ( ! empty( $list ) ) {
			$rules[] = array(
				'condition' => 'is',
				'type'      => 'field_value',
				'value'     => $list,
			);
		}

		$conditional_logic = array(
			'enabled' => ! empty( $rules ),
			'action'  => 'allow',
			'type'    => 'all',
			'rules'   => $rules,
		);

		if ( 'leave' === $action ) {
			$conditional_logic['custom_type'] = 'automation_rule_remove_from_lists';
		}

		return $conditional_logic;
	}

	/**
	 * Fetches tag conditional logic.
	 *
	 * @param \Noptin_Automated_Email $email
	 */
	protected function subscriber_tag( $email ) {

		$tag    = $email->get( 'subscriber_tag' );
		$action = $email->get( 'tag_action' );
		$rules  = array();

		if ( ! empty( $tag ) ) {
			$rules[] = array(
				'condition' => 'is',
				'type'      => 'field_value',
				'value'     => $tag,
			);
		}

		$conditional_logic = array(
			'enabled' => ! empty( $rules ),
			'action'  => 'allow',
			'type'    => 'all',
			'rules'   => $rules,
		);

		if ( 'untag' === $action ) {
			$conditional_logic['custom_type'] = 'automation_rule_remove_from_tags';
		}

		return $conditional_logic;
	}

	/**
	 * Migrates all emails saved to newsletter subscription forms.
	 *
	 */
	protected function migrate_form_welcome_emails() {
		$forms = get_noptin_optin_forms();

		// Loop through all forms.
		foreach ( $forms as $form ) {

			if (
				// Form does not exist.
				! $form->exists()

				// It is a legacy form.
				|| empty( $form->email )

				// Welcome emails have been disabled temporarily.
				|| empty( $form->email['enable_email'] )

				// subject is missing.
				|| empty( $form->email['subject'] )

				// Content is missing.
				|| empty( $form->email['content'] )
			) {
				continue;
			}

			// Create new email.
			$email = new \Noptin_Automated_Email( 'automation_rule_new_subscriber' );

			$email->name    = $form->title;
			$email->options = array(
				'subject'        => $form->email['subject'],
				'email_type'     => 'normal',
				'recipients'     => '[[email]]',
				'content_normal' => $form->email['content'],
			);

			// Save email.
			$email->save();

			if ( ! $email->exists() ) {
				continue;
			}

			// Create automation rule.
			$automation_rule = noptin_get_automation_rule();

			if ( is_wp_error( $automation_rule ) ) {
				continue;
			}

			$automation_rule->set_trigger_id( $email->get_trigger() );
			$automation_rule->set_action_id( 'email' );

			$automation_rule->set_action_settings(
				array(
					'automated_email_id' => $email->id,
				)
			);

			$automation_rule->set_trigger_settings(
				array(
					'conditional_logic' => array(
						'enabled' => true,
						'action'  => 'allow',
						'type'    => 'all',
						'rules'   => array(
							array(
								'condition' => 'is',
								'type'      => 'source',
								'value'     => $form->id,
							),
						),
					),
				)
			);

			$automation_rule->save();

			if ( $automation_rule->exists() ) {
				$email->options['automation_rule'] = $automation_rule->get_id();
				$email->save();
			}
		}
	}

	/**
	 * Migrates list emails and automation rules.
	 *
	 */
	protected function migrate_lists() {
		global $wpdb;

		// Abort if noptin_lists table does not exist.
		if ( ! $wpdb->get_var( "SHOW TABLES LIKE '{$wpdb->prefix}noptin_lists'" ) ) {
			return;
		}

		$this->migrate_tags();

		$lists = $wpdb->get_results( "SELECT * FROM {$wpdb->prefix}noptin_lists", ARRAY_A );

		// Fetch custom fields.
		$custom_fields = get_noptin_custom_fields();
		$new_fields    = array();

		// Loop through all custom fields.
		foreach ( $custom_fields as $field ) {
			if ( 'noptin_lists' !== $field['merge_tag'] && 'noptin_tags' !== $field['merge_tag'] ) {
				$new_fields[] = $field;
			}
		}

		// Abort if we have no lists to migrate.
		if ( empty( $lists ) ) {
			update_noptin_option( 'custom_fields', $new_fields );
			return;
		}

		// If we have lists, create 2 multi checkbox fields, 1 for private lists and 1 for public lists.
		$private_lists = wp_list_pluck( wp_list_filter( $lists, array( 'type' => '0' ) ), 'name', 'id' );
		$public_lists  = wp_list_pluck( wp_list_filter( $lists, array( 'type' => '1' ) ), 'name', 'id' );
		$default_list  = get_noptin_option( 'default_subscriber_lists', '' );

		if ( ! empty( $private_lists ) ) {

			// Add private lists field.
			$new_fields[] = array(
				'type'          => 'multi_checkbox',
				'merge_tag'     => 'private_lists',
				'label'         => __( 'Private Lists', 'noptin-addons-pack' ),
				'visible'       => false,
				'predefined'    => false,
				'required'      => false,
				'options'       => $this->options_to_string( $private_lists ),
				'default_value' => isset( $private_lists[ $default_list ] ) ? $default_list : '',
			);

			// Replace all meta keys in subscriber meta table.
			// where meta_key = list and value in private list IDs.
			// to private_lists.
			$ids = implode( ',', map_deep( array_keys( $private_lists ), 'absint' ) );
			$wpdb->query(
				$wpdb->prepare(
					"UPDATE {$wpdb->prefix}noptin_subscriber_meta SET meta_key = %s WHERE meta_key = %s AND meta_value IN ( $ids )", // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
					'private_lists',
					'list'
				)
			);
		}

		if ( ! empty( $public_lists ) ) {
			$new_fields[] = array(
				'type'          => 'multi_checkbox',
				'merge_tag'     => 'lists',
				'label'         => __( 'Lists', 'noptin-addons-pack' ),
				'visible'       => true,
				'predefined'    => false,
				'required'      => false,
				'options'       => $this->options_to_string( $public_lists ),
				'default_value' => ! isset( $private_lists[ $default_list ] ) ? $default_list : '',
			);

			// Convert remaining list meta keys to lists.
			$wpdb->query(
				$wpdb->prepare(
					"UPDATE {$wpdb->prefix}noptin_subscriber_meta SET meta_key = %s WHERE meta_key = %s",
					'lists',
					'list'
				)
			);
		}

		// Update custom fields.
		update_noptin_option( 'custom_fields', $new_fields );

		// Migrate list emails.
		$this->migrate_list_emails( $private_lists, $public_lists );

		// Migrate list options.
		$this->migrate_list_options( $private_lists );

		// Migrate list automation rules.
		$this->migrate_list_automation_rules( $public_lists, $private_lists );
	}

	/**
	 * Migrates list emails.
	 *
	 * @param array $private_lists Private lists.
	 * @param array $public_lists  Public lists.
	 */
	protected function migrate_list_emails( $private_lists, $public_lists ) {
		global $noptin_addons_pack_migrate_emails;

		// Abort if no emails.
		if ( empty( $noptin_addons_pack_migrate_emails ) ) {
			return;
		}

		// Loop through all emails.
		foreach ( $noptin_addons_pack_migrate_emails as $email ) {

			if ( 'automation' === get_post_meta( $email->ID, 'campaign_type', true ) ) {
				$email = new \Noptin_Automated_Email( $email->ID );
			} else {
				$email = new \Noptin_Newsletter_Email( $email->ID );
			}

			$filter_type  = $email->get( 'noptin_lists_filter_type' );
			$filter_lists = wp_parse_list( $email->get( 'noptin_lists_filter_lists' ) );

			if ( empty( $filter_type ) || empty( $filter_lists ) ) {
				continue;
			}

			$is_include     = 'in_list' === $filter_type;
			$filter_private = array_intersect( $filter_lists, array_keys( $private_lists ) );
			$filter_public  = array_intersect( $filter_lists, array_keys( $public_lists ) );

			if ( ! empty( $filter_private ) ) {
				$email->options['noptin_subscriber_options'][ $is_include ? 'private_lists' : 'private_lists_not' ] = $filter_private;
			}

			if ( ! empty( $filter_public ) ) {
				$email->options['noptin_subscriber_options'][ $is_include ? 'lists' : 'lists_not' ] = $filter_public;
			}

			unset( $email->options['noptin_lists_filter_type'] );
			unset( $email->options['noptin_lists_filter_lists'] );

			$email->save();
		}

	}

	/**
	 * Migrates list options.
	 *
	 * @param array $private_lists Private lists.
	 */
	protected function migrate_list_options( $private_lists ) {
		$new_options = array();

		foreach ( get_noptin_options() as $key => $value ) {

			// Check if key starts with noptin_lists_.
			if ( 0 === strpos( $key, 'noptin_lists_' ) ) {
				// Change key from noptin_lists_{$subscribed_via}_subscriber_lists
				// to either {$subscribed_via}_default_lists or
				// {$subscribed_via}_default_private_lists.

				// Remove leading noptin_lists_.
				$key = str_replace( 'noptin_lists_', '', $key );

				// Remove _subscriber_lists suffix.
				$key = str_replace( '_subscriber_lists', '', $key );

				// Add _default_ suffix.
				$key .= '_default_';

				if ( isset( $private_lists[ $value ] ) ) {
					$key .= 'private_';
				}

				$key .= 'lists';

				if ( '-1' === $value ) {
					$value = '';
				}
			}

			$new_options[ $key ] = $value;
		}

		update_noptin_options( $new_options );

	}

	/**
	 * Migrates automation rules.
	 *
	 * @param array $public_lists Public lists.
	 * @param array $private_lists Private lists.
	 */
	protected function migrate_list_automation_rules( $public_lists, $private_lists ) {

		/** @var \Hizzle\Noptin\DB\Automation_Rule[] $rules */
		$rules = noptin_get_automation_rules(
			array(
				'number'    => -1,
				'action_id' => array( 'add-list', 'remove-list' ),
			)
		);

		foreach ( $rules as $rule ) {

			$settings  = $rule->get_action_settings();
			$list      = isset( $settings['list'] ) ? $settings['list'] : '';
			$is_public = empty( $private_lists ) || isset( $public_lists[ $list ] );
			$field     = $is_public ? 'lists' : 'private_lists';

			if ( $rule->get_action_id() === 'add-list' ) {
				$rule->set_action_id( 'add_to_' . $field );
			} elseif ( $rule->get_action_id() === 'remove-list' ) {
				$rule->set_action_id( 'remove_from_' . $field );
			}

			$settings[ $field ] = $list;
			unset( $settings['list'] );

			$rule->set_action_settings( $settings );
			$rule->save();
		}

		// Migrate by trigger ID.
		/** @var \Hizzle\Noptin\DB\Automation_Rule[] $rules */
		$rules = noptin_get_automation_rules(
			array(
				'number'     => -1,
				'trigger_id' => array( 'add-list', 'remove-list' ),
			)
		);

		foreach ( $rules as $rule ) {

			$is_public = ! empty( $public_lists );

			// Walk conditional logic and replace tag with field_value.
			$settings = $rule->get_trigger_settings();

			if ( isset( $settings['conditional_logic']['rules'] ) ) {
				foreach ( $settings['conditional_logic']['rules'] as $index => $condition_rule ) {
					if ( 'list.id' === $condition_rule['type'] ) {
						$settings['conditional_logic']['rules'][ $index ]['type'] = 'field_value';
					}
					if ( 'list.name' === $condition_rule['type'] ) {
						$settings['conditional_logic']['rules'][ $index ]['type'] = 'field_name';
					}

					if ( isset( $private_lists[ $settings['conditional_logic']['rules'][ $index ]['value'] ] ) ) {
						$is_public = false;
					}
				}
			}

			$field = $is_public ? 'lists' : 'private_lists';
			if ( $rule->get_trigger_id() === 'add-list' ) {
				$rule->set_trigger_id( 'add_to_' . $field );
			} elseif ( $rule->get_trigger_id() === 'remove-list' ) {
				$rule->set_trigger_id( 'remove_from_' . $field );
			}

			$rule->set_trigger_settings( $settings );
			$rule->save();
		}
	}

	/**
	 * Converts an array of options to a string.
	 *
	 */
	private function options_to_string( $options ) {
		$options_string = '';

		// Key and value is separated by a pipe.
		// Each option is separated by a new line.
		foreach ( $options as $value => $label ) {
			$label           = str_replace( '|', '&#124;', $label );
			$options_string .= "{$value}|{$label}\n";
		}

		return $options_string;
	}

	/**
	 * Migrates tag emails and automation rules.
	 *
	 */
	protected function migrate_tags() {
		global $wpdb;

		// Convert remaining list meta keys to lists.
		$wpdb->query(
			$wpdb->prepare(
				"UPDATE {$wpdb->prefix}noptin_subscriber_meta SET meta_key = %s WHERE meta_key = %s",
				'tags',
				'_apTag'
			)
		);

		// Migrate tag emails.
		$this->migrate_tag_emails();

		// Migrate tag options.
		$this->migrate_tag_options();

		// Migrate tag automation rules.
		$this->migrate_tag_automation_rules();
	}

	/**
	 * Migrates tag emails.
	 *
	 */
	protected function migrate_tag_emails() {
		global $noptin_addons_pack_migrate_emails;

		// Abort if no emails.
		if ( empty( $noptin_addons_pack_migrate_emails ) ) {
			return;
		}

		// Loop through all emails.
		foreach ( $noptin_addons_pack_migrate_emails as $email ) {

			if ( 'automation' === get_post_meta( $email->ID, 'campaign_type', true ) ) {
				$email = new \Noptin_Automated_Email( $email->ID );
			} else {
				$email = new \Noptin_Newsletter_Email( $email->ID );
			}

			$filter_type = $email->get( 'noptin_tags_filter_type' );
			$filter_tags = noptin_parse_list( $email->get( 'addons_pack_tag_filter_tags' ), true );

			if ( empty( $filter_type ) || empty( $filter_tags ) ) {
				continue;
			}

			$filter_type = 'tagged' === $filter_type ? 'tags' : 'tags_not';

			$email->options['noptin_subscriber_options'][ $filter_type ] = $filter_tags;

			unset( $email->options['noptin_tags_filter_type'] );
			unset( $email->options['addons_pack_tag_filter_tags'] );
			$email->save();
		}

	}

	/**
	 * Migrates tag options.
	 *
	 */
	protected function migrate_tag_options() {
		$new_options = array();

		foreach ( get_noptin_options() as $key => $value ) {

			// Check if key ends with _subscriber_tags.
			if ( '_subscriber_tags' === substr( $key, -16 ) ) {

				// Change key from noptin_addons_pack_{$subscribed_via}_subscriber_tags
				// to {$subscribed_via}_default_tags.
				// Remove leading noptin_addons_pack_.
				$key = str_replace( 'noptin_addons_pack_', '', $key );

				// Remove _subscriber_tags suffix.
				$key = str_replace( '_subscriber_tags', '', $key );

				// Add _default_tags suffix.
				$key .= '_default_tags';
			}

			$new_options[ $key ] = $value;
		}

		update_noptin_options( $new_options );

	}

	/**
	 * Migrates automation rules.
	 *
	 */
	protected function migrate_tag_automation_rules() {

		// Migrate by action ID.
		/** @var \Hizzle\Noptin\DB\Automation_Rule[] $rules */
		$rules = noptin_get_automation_rules(
			array(
				'number'    => -1,
				'action_id' => array( 'add-tag', 'remove-tag' ),
			)
		);

		foreach ( $rules as $rule ) {

			if ( $rule->get_action_id() === 'add-tag' ) {
				$rule->set_action_id( 'add_to_tags' );
			} elseif ( $rule->get_action_id() === 'remove-tag' ) {
				$rule->set_action_id( 'remove_from_tags' );
			}

			$settings         = $rule->get_action_settings();
			$settings['tags'] = empty( $settings['tag'] ) ? array() : $settings['tag'];

			$rule->set_action_settings( $settings );
			$rule->save();
		}

		// Migrate by trigger ID.
		/** @var \Hizzle\Noptin\DB\Automation_Rule[] $rules */
		$rules = noptin_get_automation_rules(
			array(
				'number'     => -1,
				'trigger_id' => array( 'add-tag', 'remove-tag' ),
			)
		);

		foreach ( $rules as $rule ) {

			if ( $rule->get_trigger_id() === 'add-tag' ) {
				$rule->set_trigger_id( 'add_to_tags' );
			} elseif ( $rule->get_trigger_id() === 'remove-tag' ) {
				$rule->set_trigger_id( 'remove_from_tags' );
			}

			// Walk conditional logic and replace tag with field_value.
			$settings = $rule->get_trigger_settings();

			if ( isset( $settings['conditional_logic']['rules'] ) ) {
				foreach ( $settings['conditional_logic']['rules'] as $index => $condition_rule ) {
					if ( 'tag' === $condition_rule['type'] ) {
						$settings['conditional_logic']['rules'][ $index ]['type'] = 'field_value';
					}
				}
			}

			$rule->set_trigger_settings( $settings );
			$rule->save();
		}
	}
}
