<?php

namespace Noptin\Addons_Pack\Schedule_Rules;

/**
 * Schedules rules to run at a later time.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Schedules rules to run at a later time.
 *
 */
class Main {

	/**
	 * Stores the main component instance.
	 *
	 * @access private
	 * @var    Main $instance The main component instance.
	 * @since  1.0.0
	 */
	private static $instance = null;

	/**
	 * Get active instance
	 *
	 * @access public
	 * @since  1.0.0
	 * @return Main The main component instance.
	 */
	public static function instance() {

		if ( empty( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Class Constructor.
	 */
	public function __construct() {
		add_action( 'noptin_delay_automation_rule_execution', array( $this, 'delay_automation_rule' ), 10, 3 );
		add_action( 'noptin_run_delayed_automation_rule', array( $this, 'run_delayed_automation_rule' ), 10, 2 );
	}

	/**
	 * Delays an automation rule.
	 *
	 * @param \Noptin_Automation_Rule $rule The rule to delay.
	 * @param array $args The rule arguments.
	 * @param int $delay The delay in seconds.
	 */
	public function delay_automation_rule( $rule, $args, $delay ) {

		// Fetch the trigger.
		$trigger = noptin()->automation_rules->get_trigger( $rule->trigger_id );

		// Abort if it does not exist.
		if ( empty( $trigger ) ) {
			return;
		}

		// Serialize the trigger arguments.
		$args = $trigger->serialize_trigger_args( $args );

		do_action(
			'noptin_tasks_schedule',
			'noptin_run_delayed_automation_rule',
			array(
				'rule_id' => $rule->id,
				'args'    => $args,
			),
			$delay
		);
	}

	/**
	 * Runs a delayed automation rule.
	 *
	 * @param string $automation_rule The rule id to schedule.
	 * @param array $args The trigger arguments.
	 */
	public function run_delayed_automation_rule( $rule_id, $args ) {

		// Fetch the automation rule.
		$rule = new \Noptin_Automation_Rule( $rule_id );

		if ( ! $rule->exists() ) {
			throw new \Exception( 'Automation rule not found' );
		}

		// Fetch the trigger.
		$trigger = noptin()->automation_rules->get_trigger( $rule->trigger_id );

		if ( empty( $trigger ) ) {
			throw new \Exception( 'Invalid or unregistered trigger' );
		}

		// Fetch the action.
		$action = noptin()->automation_rules->get_action( $rule->action_id );

		if ( empty( $action ) ) {
			throw new \Exception( 'Invalid or unregistered action' );
		}

		// Unserialize the trigger arguments.
		$args = $trigger->unserialize_trigger_args( $args );

		// Abort if the trigger does not support scheduling.
		if ( ! is_array( $args ) ) {
			throw new \Exception( 'Trigger does not support scheduling' );
		}

		// Set the current email.
		$GLOBALS['current_noptin_email'] = $trigger->get_subject_email( $args['subject'], $rule, $args );

		// Ensure that the rule is valid for the provided args.
		if ( $trigger->is_rule_valid_for_args( $rule, $args, $args['subject'], $action ) ) {
			$action->maybe_run( $args['subject'], $rule, $args );
		} else {
			throw new \Exception( 'Automation rule is no longer valid for the provided arguments' );
		}
	}

}
