<?php

namespace Noptin\Addons_Pack\Coupon_Codes;

/**
 * Adds dynamic coupons functionality.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Adds dynamic coupons functionality.
 *
 */
class Main {

	/**
	 * Stores the main component instance.
	 *
	 * @access private
	 * @var    Main $instance The main component instance.
	 * @since  1.0.0
	 */
	private static $instance = null;

	/**
	 * Get active instance
	 *
	 * @access public
	 * @since  1.0.0
	 * @return Main The main component instance.
	 */
	public static function instance() {

		if ( empty( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Class Constructor.
	 */
	private function __construct() {

        // Register settings tab.
		add_filter( 'noptin_get_setting_sections', array( $this, 'filter_setting_tabs' ) );
		add_filter( 'noptin_get_settings', array( $this, 'filter_settings' ) );

		// Delete discount codes.
		add_action( 'noptin_daily_maintenance', array( $this, 'delete_expired_coupon_codes' ) );

        // Register merge tag.
		add_filter( 'noptin_automation_rules_load', array( $this, 'add_trigger' ) );
        add_filter( 'noptin_automation_trigger_known_smart_tags', array( $this, 'register_automation_trigger_smart_tags' ) );
        add_action( 'noptin_email_manager_init', array( $this, 'register_merge_tags' ) );
        add_action( 'init_noptin_form_manager', array( $this, 'register_merge_tags' ) );

        // Register shortcode.
        add_shortcode( 'noptin_coupon_code', array( $this, 'handle_shortcode' ) );
	}

    /**
	 * Filters setting tabs.
	 *
	 * @param array $tabs
	 */
	public function filter_setting_tabs( $tabs ) {
		$tabs['coupon_codes'] = __( 'Coupon Codes', 'noptin-addons-pack' );
		return $tabs;
	}

    /**
	 * Filters settings.
	 *
	 * @param array $settings
	 */
	public function filter_settings( $settings ) {

		$has_supported_integration = false;

		$settings['coupon_codes_hero'] = array(
			'el'      => 'hero',
			'section' => 'coupon_codes',
			'content' => __( 'Coupon Codes', 'noptin-addons-pack' ),
		);

		// WooCommerce integration.
		if ( class_exists( 'WooCommerce' ) ) {

			$settings['wc_coupon_template'] = array(
				'el'          => 'input',
				'type'        => 'text',
				'section'     => 'coupon_codes',
				'label'       => __( 'WooCommerce Template', 'noptin-addons-pack' ),
				'placeholder' => __( 'WooCommerce Coupon Code', 'noptin-addons-pack' ),
				'description' => __( 'Enter an existing WooCommerce coupon code (case sensitive) to use as a coupon template. New coupons will be created based on the provided coupon code.', 'noptin-addons-pack' ),
			);

			$has_supported_integration = true;
			// Link to the documentation for the integration.
		}

		// GetPaid integration.
		if ( did_action( 'getpaid_actions' ) ) {

			$settings['getpaid_coupon_template'] = array(
				'el'          => 'input',
				'type'        => 'text',
				'section'     => 'coupon_codes',
				'label'       => __( 'GetPaid Template', 'noptin-addons-pack' ),
				'placeholder' => __( 'GetPaid Discount Code', 'noptin-addons-pack' ),
				'description' => __( 'Enter an existing GetPaid discount code to use as a discount template. New discount codes will be created based on the provided discount code.', 'noptin-addons-pack' ),
			);

			$has_supported_integration = true;
		}

		// EDD integration.
		if ( class_exists( 'Easy_Digital_Downloads' ) ) {

			$settings['edd_coupon_template'] = array(
				'el'          => 'input',
				'type'        => 'text',
				'section'     => 'coupon_codes',
				'label'       => __( 'EDD Template', 'noptin-addons-pack' ),
				'placeholder' => __( 'EDD Discount Code', 'noptin-addons-pack' ),
				'description' => __( 'Enter an existing EDD discount code to use as a discount template. New discount codes will be created based on the provided discount code.', 'noptin-addons-pack' ),
			);

			$has_supported_integration = true;
		}

		if ( $has_supported_integration ) {

			$settings['coupon_code_length'] = array(
				'el'          => 'input',
				'type'        => 'number',
				'section'     => 'coupon_codes',
				'label'       => __( 'Coupon Length', 'noptin-addons-pack' ),
				'placeholder' => '8',
				'default'     => '8',
				'description' => __( 'The number of characters for the auto-generated coupon code.', 'noptin-addons-pack' ),
			);

			$settings['coupon_code_prefix'] = array(
				'el'          => 'input',
				'type'        => 'text',
				'section'     => 'coupon_codes',
				'placeholder' => __( 'For example, DIS-', 'noptin-addons-pack' ),
				'label'       => __( 'Coupon Code Prefix', 'noptin-addons-pack' ),
				'description' => __( 'Optional. Enter text to prepend to the auto-generated coupon code.', 'noptin-addons-pack' ),
			);

			$settings['coupon_code_suffix'] = array(
				'el'          => 'input',
				'type'        => 'text',
				'section'     => 'coupon_codes',
				'label'       => __( 'Coupon Code Suffix', 'noptin-addons-pack' ),
				'description' => __( 'Optional. Enter text to append to the auto-generated coupon code.', 'noptin-addons-pack' ),
			);

			$settings['coupon_code_lifetime'] = array(
				'el'          => 'input',
				'type'        => 'number',
				'section'     => 'coupon_codes',
				'label'       => __( 'Coupon Expiry', 'noptin-addons-pack' ),
				'placeholder' => __( 'Never Expire', 'noptin-addons-pack' ),
				'description' => __( 'Optional. Enter the number of seconds after which a coupon code will expire.', 'noptin-addons-pack' ),
			);

		}

		// Let users specify a comma separated list of existing coupon codes.
		$settings['coupon_codes_presaved'] = array(
			'el'          => 'textarea',
			'section'     => 'coupon_codes',
			'label'       => __( 'Coupon Codes', 'noptin-addons-pack' ),
			'placeholder' => 'DIS-1234,DIS-5678,DIS-9012',
			'description' => __( 'Optional. If provided, users will be sent a random coupon code from these coupon codes instead of generating new coupon codes.', 'noptin-addons-pack' ),
		);

		return $settings;
	}

	/**
	 * Loads automation rule triggers.
	 *
	 * @param \Noptin_Automation_Rules $rules
	 */
	public function add_trigger( $rules ) {
        $rules->add_trigger( new Used_Coupon_Trigger() );
	}

    /**
	 * Registers known automation trigger smart tags.
	 *
	 * @param array $tags The known tags.
	 * @return array
	 */
	public function register_automation_trigger_smart_tags( $tags ) {

		$tags['noptin_coupon_code'] = array(
			'description' => __( 'Unique Coupon Code', 'noptin-addons-pack' ),
			'callback'    => array( $this, 'handle_merge_tag' ),
			'example'     => 'noptin_coupon_code',
		);

		return $tags;
	}

    /**
	 * Adds the coupon codes merge tag to emails and forms.
	 *
	 * @param \Noptin_Email_Manager|\Noptin_Form_Manager $manager
	 */
	public function register_merge_tags( $manager ) {
        $manager->tags->add_tag(
			'noptin_coupon_code',
			array(
				'description' => __( 'Unique Coupon Code', 'noptin-addons-pack' ),
				'callback'    => array( $this, 'handle_merge_tag' ),
				'example'     => 'noptin_coupon_code',
			)
		);
	}

    /**
	 * Saves a WooCommerce Coupon template.
	 *
	 * @since       1.0.0
	 * @param string $coupon_code
	 * @param null|float $value
	 * @param false|int $expires
	 * @param string|false $coupon_key
	 * @param string|false $template_code
	 */
	public function save_wc_coupon_code( $coupon_code, $value, $expires, $coupon_key = false, $template_code = false ) {
		global $wpdb;

		$template = $this->get_wc_template( $template_code );

		if ( empty( $template ) ) {
			return;
		}

		try {

			$new_coupon = new \WC_Coupon();
			$new_coupon->set_code( $coupon_code );
			$new_coupon->set_description( $template->get_description() );
			$new_coupon->set_discount_type( $template->get_discount_type() );
			$new_coupon->set_amount( empty( $value ) ? $template->get_amount() : $value );
			$new_coupon->set_date_created( time() );
			$new_coupon->set_individual_use( $template->get_individual_use() );
			$new_coupon->set_product_ids( $template->get_product_ids() );
			$new_coupon->set_excluded_product_ids( $template->get_excluded_product_ids() );
			$new_coupon->set_usage_limit( $template->get_usage_limit() );
			$new_coupon->set_usage_limit_per_user( $template->get_usage_limit_per_user() );
			$new_coupon->set_limit_usage_to_x_items( $template->get_limit_usage_to_x_items() );
			$new_coupon->set_free_shipping( $template->get_free_shipping() );
			$new_coupon->set_product_categories( $template->get_product_categories() );
			$new_coupon->set_excluded_product_categories( $template->get_excluded_product_categories() );
			$new_coupon->set_exclude_sale_items( $template->get_exclude_sale_items() );
			$new_coupon->set_minimum_amount( $template->get_minimum_amount() );
			$new_coupon->set_maximum_amount( $template->get_maximum_amount() );

			if ( ! empty( $expires ) ) {

				// WC expects a UTC timestring.
				$new_coupon->set_date_expires( $expires );
				$new_coupon->add_meta_data( 'noptin_delete_coupon_on', gmdate( 'Y-m-d H:i:s', $expires ) );
			}

			if ( ! empty( $coupon_key ) ) {
				$new_coupon->add_meta_data( 'noptin_coupon', $coupon_key );
			}

			$new_coupon->add_meta_data( 'is_noptin_coupon', 1 );
			$new_coupon->save();

			// Advanced coupon codes for Woo compatibility.
			if ( $new_coupon->get_id() ) {
				$previous_meta = get_post_meta( $template->get_id() );
				$cloned_meta   = get_post_meta( $new_coupon->get_id() );
				$new_coupon_id = $new_coupon->get_id();

				$to_save = array_filter(
					$previous_meta,
					function ( $key ) use ( $cloned_meta ) {
						return ! isset( $cloned_meta[ $key ] ) && '_used_by' !== $key;
					},
					ARRAY_FILTER_USE_KEY
				);

				if ( is_array( $to_save ) && ! empty( $to_save ) ) {

					$query = "INSERT INTO $wpdb->postmeta ( post_id , meta_key , meta_value ) VALUES";
					$first = true;

					foreach ( $to_save as $key => $value ) {

						if ( ! $first ) {
							$query .= ',';
						}

						if ( count( $value ) === 1 ) {

							$clean_value = esc_sql( $value[0] );
							$query      .= " ( $new_coupon_id , '$key' , '$clean_value' )";

						} else {

							$temp = array_map(
								function ( $v ) use ( $new_coupon_id, $key ) {

									$clean_value = esc_sql( $v );
									return " ( $new_coupon_id , '$key' , '$clean_value' )";

								},
								$value
							);

							$query .= implode( ',', $temp );
						}

						$first = false;
					}

					$wpdb->query( $query ); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared

				}

				if ( taxonomy_exists( 'shop_coupon_cat' ) ) {
					$categories = wp_get_post_terms( $template->get_id(), 'shop_coupon_cat', array( 'fields' => 'ids' ) );

					if ( ! is_wp_error( $categories ) && ! empty( $categories ) ) {
						wp_set_post_terms( $new_coupon->get_id(), $categories, 'shop_coupon_cat' );
					}
				}
			}
		} catch ( \Exception $e ) {
			log_noptin_message( $e->getMessage() );
		}

    }

    /**
	 * Retrieves a WooCommerce Coupon template.
	 *
	 * @since       1.0.0
	 * @return \WC_Coupon|false
	 * @param string|false $template_code
	 */
	public function get_wc_template( $template_code = false ) {

		if ( empty( $template_code ) ) {
			$template_code = get_noptin_option( 'wc_coupon_template' );
		}

		if ( ! empty( $template_code ) && function_exists( 'wc_get_coupon_id_by_code' ) ) {
			$coupon_id = wc_get_coupon_id_by_code( trim( $template_code ) );
		}

		if ( empty( $coupon_id ) ) {
			return false;
		}

		return new \WC_Coupon( $coupon_id );

    }

    /**
	 * Saves a GetPaid Coupon template.
	 *
	 * @since       1.0.0
	 * @param string $coupon_code
	 * @param null|float $value
	 * @param false|int $expires
	 * @param string|false $coupon_key
	 * @param string|false $template_code
	 */
	public function save_getpaid_coupon_code( $coupon_code, $value, $expires, $coupon_key = false, $template_code = false ) {

		$template = $this->get_getpaid_template( $template_code );

		if ( empty( $template ) ) {
			return;
		}

		try {

			$new_coupon = new \WPInv_Discount();
			$new_coupon->set_status( 'publish' );
			$new_coupon->set_code( $coupon_code );
			$new_coupon->set_name( $coupon_code );
			$new_coupon->set_description( $template->get_description() );
			$new_coupon->set_author( $template->get_author() );
			$new_coupon->set_type( $template->get_type() );
			$new_coupon->set_amount( empty( $value ) ? $template->get_amount() : $value );
			$new_coupon->set_date_created( current_time( 'mysql' ) );
			$new_coupon->set_max_uses( $template->get_max_uses() );
			$new_coupon->set_is_single_use( $template->get_is_single_use() );
			$new_coupon->set_items( $template->get_items() );
			$new_coupon->set_excluded_items( $template->get_excluded_items() );
			$new_coupon->set_is_recurring( $template->get_is_recurring() );
			$new_coupon->set_min_total( $template->get_min_total() );
			$new_coupon->set_maximum_total( $template->get_maximum_total() );

			if ( ! empty( $expires ) ) {
				$new_coupon->set_expiration( gmdate( 'Y-m-d H:i:s', $expires + ( (float) get_option( 'gmt_offset' ) * HOUR_IN_SECONDS ) ) );
				$new_coupon->add_meta_data( 'noptin_delete_coupon_on', gmdate( 'Y-m-d H:i:s', $expires ) );
			}

			if ( ! empty( $coupon_key ) ) {
				$new_coupon->add_meta_data( 'noptin_coupon', $coupon_key );
			}

			$new_coupon->add_meta_data( 'is_noptin_coupon', 1 );
			$new_coupon->save();

		} catch ( \Exception $e ) {
			log_noptin_message( $e->getMessage() );
		}

    }

	/**
	 * Retrieves a GetPaid Coupon template.
	 *
	 * @since       1.0.0
	 * @return \WPInv_Discount|false
	 * @param string|false $template_code
	 */
	public function get_getpaid_template( $template_code = false ) {

		if ( empty( $template_code ) ) {
			$template_code = get_noptin_option( 'getpaid_coupon_template' );
		}

		if ( ! empty( $template_code ) && function_exists( 'wpinv_get_discount_by' ) ) {
			return wpinv_get_discount_by( null, trim( $template_code ) );
		}

		return false;

    }

    /**
	 * Saves an EDD Coupon template.
	 *
	 * @since       1.0.0
	 * @param string $coupon_code
	 * @param null|float $value
	 * @param false|int $expires
	 * @param string|false $coupon_key
	 * @param string|false $template_code
	 */
	public function save_edd_coupon_code( $coupon_code, $value, $expires, $coupon_key = false, $template_code = false ) {

		$template = $this->get_edd_template( $template_code );

		if ( empty( $template ) ) {
			return;
		}

		try {

			$new_coupon                    = new \EDD_Discount();
			$new_coupon->name              = $template->get_name();
			$new_coupon->code              = $coupon_code;
			$new_coupon->status            = 'active';
			$new_coupon->type              = $template->get_type();
			$new_coupon->amount            = empty( $value ) ? $template->get_amount() : $value;
			$new_coupon->product_reqs      = $template->get_product_reqs();
			$new_coupon->excluded_products = $template->get_excluded_products();
			$new_coupon->start             = $template->get_start();
			$new_coupon->max_uses          = $template->get_max_uses();
			$new_coupon->min_price         = $template->get_min_price();
			$new_coupon->is_single_use     = $template->get_is_single_use();
			$new_coupon->is_not_global     = $template->get_is_not_global();
			$new_coupon->product_condition = $template->get_product_condition();

			if ( ! empty( $expires ) ) {
				$new_coupon->expiration = gmdate( 'Y-m-d H:i:s', $expires + ( (float) get_option( 'gmt_offset' ) * HOUR_IN_SECONDS ) );
			}

			$new_coupon->save();

			if ( ! empty( $expires ) ) {
				update_post_meta( $new_coupon->get_ID(), 'noptin_delete_coupon_on', gmdate( 'Y-m-d H:i:s', $expires ) );
			}

			if ( ! empty( $coupon_key ) ) {
				update_post_meta( $new_coupon->get_ID(), 'noptin_coupon', $coupon_key );
			}

			update_post_meta( $new_coupon->get_ID(), 'is_noptin_coupon', 1 );
		} catch ( \Exception $e ) {
			log_noptin_message( $e->getMessage() );
		}

    }

	/**
	 * Retrieves an EDD Coupon template.
	 *
	 * @since       1.0.0
	 * @param string|false $template_code
	 * @return \EDD_Discount|false
	 */
	public function get_edd_template( $template_code = false ) {

		if ( empty( $template_code ) ) {
			$template_code = get_noptin_option( 'edd_coupon_template' );
		}

		if ( ! empty( $template_code ) && class_exists( 'EDD_Discount' ) ) {
			$discount = new \EDD_Discount( trim( $template_code ), true );
		}

		return empty( $discount ) || empty( $discount->ID ) ? false : $discount;

    }

    /**
	 * Retrieves the discounts expiry date as a timestamp.
	 *
	 * @since       1.0.0
	 * @return int|false
	 */
	public function get_expires_date() {
		$expires = absint( get_noptin_option( 'coupon_code_lifetime' ) );
		return empty( $expires ) ? false : time() + $expires;
    }

    /**
	 * Generates a random coupon code.
	 *
	 * @since       1.0.0
	 * @return int|false
	 */
	public function generate_random_coupon_code() {

		$chars  = apply_filters( 'noptin_coupon_code_generator_characters', 'ABCDEFGHJKMNPQRSTUVWXYZ23456789' );
		$length = apply_filters( 'noptin_coupon_code_generator_character_length', (int) get_noptin_option( 'coupon_code_length' ) );
		$prefix = apply_filters( 'noptin_coupon_code_generator_prefix', get_noptin_option( 'coupon_code_prefix' ) );
		$suffix = apply_filters( 'noptin_coupon_code_generator_suffix', get_noptin_option( 'coupon_code_suffix' ) );

		if ( empty( $length ) ) {
			$length = 8;
		}

		$coupon = $prefix;
		for ( $i = 0; $i < $length; $i++ ) {
			$coupon .= substr( $chars, wp_rand( 0, strlen( $chars ) - 1 ), 1 );
		}
		$coupon .= $suffix;

		return apply_filters( 'noptin_coupon_code_generator_random_code', $coupon, $length, $chars, $prefix, $suffix );

    }

	/**
	 * Handles the coupon code shortcode.
	 *
	 * @since       1.10.1
	 * @param array $atts
	 * @return string
	 */
	public function handle_shortcode( $atts = array() ) {

		$atts = shortcode_atts(
			array(
				'value'    => '',
				'expires'  => '',
				'template' => '',
			),
			$atts,
			'noptin_coupon_code'
		);

		return $this->handle_merge_tag( array_filter( $atts ) );

	}

    /**
	 * Generates a unique coupon code for the recipient.
	 *
	 * @since 1.0.0
	 */
	public function handle_merge_tag( $args = array() ) {
		global $generate_noptin_coupon_codes;

		$coupon_key = $this->get_coupon_key( $args );

		if ( empty( $coupon_key ) ) {
			return $this->generate_coupon_code( $args );
		}

		if ( ! is_array( $generate_noptin_coupon_codes ) ) {
			$generate_noptin_coupon_codes = array();
		} elseif ( isset( $generate_noptin_coupon_codes[ $coupon_key ] ) ) {
			return $generate_noptin_coupon_codes[ $coupon_key ];
		}

		$generate_noptin_coupon_codes[ $coupon_key ] = $this->generate_coupon_code( $args );
		return $generate_noptin_coupon_codes[ $coupon_key ];
	}

	/**
	 * Generates a unique coupon code for the recipient.
	 *
	 * @since 1.0.0
	 */
	public function generate_coupon_code( $args = array() ) {

		$presaved_codes = get_noptin_option( 'coupon_codes_presaved' );

		if ( ! empty( $presaved_codes ) ) {
			$presaved_codes = noptin_parse_list( $presaved_codes );

			// Return a random coupon code.
			if ( ! empty( $presaved_codes ) ) {
				return $presaved_codes[ array_rand( $presaved_codes ) ];
			}
		}

		// Fetch saved coupon code.
		$coupon_code = $this->get_saved_coupon_code( $args );

		if ( ! empty( $coupon_code ) ) {
			return $coupon_code;
		}

		// Prepare coupon value and cache key.
		$value      = empty( $args['value'] ) ? null : (float) $args['value'];
		$coupon_key = $this->get_coupon_key( $args );

		// Generate coupon code.
		$coupon_code = $this->generate_random_coupon_code();

		// Calculate expiry date.
		$expires = $this->get_expires_date();

		if ( ! empty( $args['expires'] ) ) {
			$expires = time() + intval( $args['expires'] );
		}

		// A custom template to use.
		$template_code = empty( $args['template'] ) ? null : $args['template'];

		// Save the coupon code.
		$this->save_wc_coupon_code( $coupon_code, $value, $expires, $coupon_key, $template_code );
		$this->save_getpaid_coupon_code( $coupon_code, $value, $expires, $coupon_key, $template_code );
		$this->save_edd_coupon_code( $coupon_code, $value, $expires, $coupon_key, $template_code );

		return $coupon_code;
	}

    /**
	 * Fetches a saved coupon code that's not yet used.
	 *
	 * @param array $args
	 * @return string|false
	 */
	public function get_saved_coupon_code( $args ) {
		global $wpdb;

		$coupon_key = $this->get_coupon_key( $args );

		if ( empty( $coupon_key ) ) {
			return false;
		}

		// Search post meta for a coupon code.
		$previous_coupons = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = 'noptin_coupon' AND meta_value = %s",
				$coupon_key
			)
		);

		// Loop through previous coupons.
		foreach ( $previous_coupons as $coupon_id ) {
			$post = get_post( $coupon_id );

			if ( empty( $post ) ) {
				continue;
			}

			switch ( $post->post_type ) {
				case 'shop_coupon':
					if ( class_exists( 'WC_Coupon' ) ) {

						$coupon = new \WC_Coupon( $post->ID );

						if ( 0 === $coupon->get_usage_count() && ( ! $coupon->get_date_expires() || time() < $coupon->get_date_expires()->getTimestamp() ) ) {
							return $coupon->get_code();
						}
					}

					break;
				case 'wpi_discount':
					if ( class_exists( 'WPInv_Discount' ) ) {

						$coupon = new \WPInv_Discount( $post->ID );

						if ( 0 === $coupon->get_uses() && ! $coupon->is_expired() ) {
							return $coupon->get_code();
						}
					}

					break;
				case 'edd_discount':
					if ( class_exists( 'EDD_Discount' ) ) {
						$coupon = new \EDD_Discount( $post->ID );

						if ( 0 === (int) $coupon->get_uses() && ! $coupon->is_expired() ) {
							return $coupon->get_code();
						}
					}

					break;
			}
		}

		return false;
	}

	/**
	 * Retrieves the current subscriber key.
	 *
	 * @return string|false
	 */
	private function get_current_subscriber_key() {

		if ( ! empty( $GLOBALS['current_noptin_email'] ) ) {
			return $GLOBALS['current_noptin_email'];
		}

		$user_data = wp_get_current_user();
		if ( ! empty( $user_data->user_email ) ) {
			$user_data->user_email;
		}

		if ( ! empty( $_GET['noptin_key'] ) ) {
			return sanitize_text_field( urldecode( $_GET['noptin_key'] ) );
		}

		if ( ! empty( $_COOKIE['noptin_email_subscribed'] ) ) {
			return sanitize_text_field( $_COOKIE['noptin_email_subscribed'] );
		}

		return false;
	}

    /**
	 * Retrieves a coupon cache key.
	 *
	 * @param array $args
	 */
	public function get_coupon_key( $args ) {

		$current_subscriber = $this->get_current_subscriber_key();

		if ( empty( $current_subscriber ) ) {
			return false;
		}

		return md5( $current_subscriber . wp_json_encode( $args ) );
	}

    /**
	 * Deletes expired coupon codes.
	 *
	 * @since 1.0.0
	 */
	public function delete_expired_coupon_codes() {
		global $wpdb;

		$expired_coupon_codes = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT post_id FROM {$wpdb->postmeta} WHERE meta_key = 'noptin_delete_coupon_on' AND CAST(meta_value AS DATETIME) < %d",
				time()
			)
		);

		foreach ( $expired_coupon_codes as $coupon_id ) {
			$post = get_post( $coupon_id );

			if ( empty( $post ) ) {
				continue;
			}

			switch ( $post->post_type ) {
				case 'shop_coupon':
					if ( class_exists( 'WC_Coupon' ) ) {

						$coupon = new \WC_Coupon( $post->ID );

						if ( 0 === $coupon->get_usage_count() && $coupon->get_date_expires() && time() > $coupon->get_date_expires()->getTimestamp() ) {
							$coupon->delete( true );
						}
					}

					break;
				case 'wpi_discount':
					if ( class_exists( 'WPInv_Discount' ) ) {

						$coupon = new \WPInv_Discount( $post->ID );

						if ( 0 === $coupon->get_uses() && $coupon->is_expired() ) {
							$coupon->delete( true );
						}
					}

					break;
				case 'edd_discount':
					if ( class_exists( 'EDD_Discount' ) ) {
						$coupon = new \EDD_Discount( $post->ID );

						if ( 0 === (int) $coupon->get_uses() && $coupon->is_expired() ) {
							wp_delete_post( $coupon->get_ID() );
						}
					}

					break;
			}
		}
    }
}
