<?php

namespace Noptin\Addons_Pack\Users\Actions;

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * Adds a user role.
 *
 * @since 1.10.0
 */
class Add_User_Role extends Abstract_Action {

	/**
	 * @inheritdoc
	 */
	public function get_id() {
		return 'add_user_role';
	}

	/**
	 * @inheritdoc
	 */
	public function get_name() {
		return __( 'User > Add Role', 'noptin-addons-pack' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_description() {
		return __( 'Adds a role to the user', 'noptin-addons-pack' );
	}

	/**
	 * Retrieve the actions's rule table description.
	 *
	 * @since 1.11.9
	 * @param Noptin_Automation_Rule $rule
	 * @return array
	 */
	public function get_rule_table_description( $rule ) {
		$settings = $rule->action_settings;

		// Ensure we have a role.
		if ( empty( $settings['role'] ) ) {
			return sprintf(
				'<span class="noptin-rule-error">%s</span>',
				esc_html__( 'Error: Role not specified', 'noptin-addons-pack' )
			);
		}

		$roles = array_map( 'translate_user_role', wp_list_pluck( array_reverse( get_editable_roles() ), 'name' ) );
		$role  = isset( $roles[ $settings['role'] ] ) ? $roles[ $settings['role'] ] : $settings['role'];
		$meta  = array(
			__( 'Role', 'noptin-addons-pack' ) => $role,
		);

		return $this->rule_action_meta( $meta, $rule );
	}

	/**
	 * @inheritdoc
	 */
	public function get_settings() {

		return array(

			'role' => array(
				'el'          => 'select',
				'label'       => __( 'Role', 'noptin-addons-pack' ),
				'placeholder' => __( 'Select a role', 'noptin-addons-pack' ),
				'default'     => 'subscriber',
				'options'     => array_map( 'translate_user_role', wp_list_pluck( array_reverse( get_editable_roles() ), 'name' ) ),
			),
		);
	}

	/**
	 * Returns whether or not the action can run (dependancies are installed).
	 *
	 * @since 1.3.3
	 * @param mixed $subject The subject.
	 * @param Noptin_Automation_Rule $rule The automation rule used to trigger the action.
	 * @param array $args Extra arguments passed to the action.
	 * @return bool
	 */
	public function can_run( $subject, $rule, $args ) {

		// Abort if we have no role.
		if ( empty( $rule->action_settings['role'] ) ) {
			return false;
		}

		return parent::can_run( $subject, $rule, $args );
	}

	/**
	 * Updates the user's role.
	 *
	 * @since 1.10.0
	 * @param mixed $subject The subject.
	 * @param Noptin_Automation_Rule $rule The automation rule used to trigger the action.
	 * @param array $args Extra arguments passed to the action.
	 * @return void
	 */
	public function run( $subject, $rule, $args ) {
		$this->process( $this->get_user( $subject, $rule, $args ), $rule->action_settings['role'] );
	}

	/**
	 * @param \WP_User $user
	 * @param string $role
	 */
	protected function process( $user, $role ) {
		// Add the user to the role.
		$user->add_role( $role );
	}
}
