<?php

namespace Noptin\WooCommerce\Subscriptions;

/**
 * Main subscriptions module class.
 *
 * @since 1.0.1
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

/**
 * The main subscriptions module class.
 */
class Main {

	/**
	 * Class constructor.
	 */
	public function __construct() {
		add_filter( 'noptin_woocommerce_sending_options', array( $this, 'add_subscription_sending_options' ) );
		add_filter( 'noptin_users_sending_options', array( $this, 'add_subscription_sending_options' ) );
		add_filter( 'noptin_subscriber_sending_options', array( $this, 'add_subscription_sending_options' ) );
		add_filter( 'noptin_woocommerce_can_email_customer_for_campaign', array( $this, 'can_email_customer' ), 10, 3 );
		add_filter( 'noptin_users_can_email_user_for_campaign', array( $this, 'can_email_user' ), 10, 3 );
		add_filter( 'noptin_subscribers_can_email_subscriber_for_campaign', array( $this, 'can_email_subscriber' ), 10, 3 );
	}

	/**
	 * Registers subscription sending options.
	 *
	 * @param array $options subscription sending options.
	 */
	public function add_subscription_sending_options( $options ) {

		// Limit to customers who have subscriptions to specific products.
		$options['subscribed'] = array(
			'label'       => __( 'Subscribed To', 'noptin-woocommerce' ),
			'type'        => 'text',
			'placeholder' => sprintf(
				// translators: %s: Example product ids.
				__( 'For example, %s', 'noptin-woocommerce' ),
				'4, 5'
			),
			'description' => __( 'Optional. Provide a comma separated list of product ids if you only want to send this email to customers who have an active subscription for specific products', 'noptin-woocommerce' ),
		);

		// Limit to customers who do not have subscriptions to specific products.
		$options['not_subscribed'] = array(
			'label'       => __( 'Not Subscribed To', 'noptin-woocommerce' ),
			'type'        => 'text',
			'placeholder' => sprintf(
				// translators: %s: Example product ids.
				__( 'For example, %s', 'noptin-woocommerce' ),
				'6, 7'
			),
			'description' => __( 'Optional. Provide a comma separated list of product ids if you only want to send this email to customers who do not have an active subscription for specific products', 'noptin-woocommerce' ),
		);

		return $options;
	}

	/**
	 * Determines whether a customer can be emailed for a campaign.
	 *
	 * @param bool   $can_email whether a customer can be emailed.
	 * @param array  $campaign_options campaign options.
	 * @param \WC_Customer $customer customer.
	 */
	public function can_email_customer( $can_email, $campaign_options, $customer ) {

		// Abort if already can't email.
		if ( ! $can_email || ( empty( $campaign_options['subscribed'] ) && empty( $campaign_options['not_subscribed'] ) ) ) {
			return $can_email;
		}

		if ( ! $customer->get_id() ) {
			return $can_email;
		}

		return $this->can_email_recipient( $customer->get_id(), $campaign_options );
	}

	/**
	 * Determines whether a user can be emailed for a campaign.
	 *
	 * @param bool     $can_email whether a user can be emailed.
	 * @param array    $campaign_options campaign options.
	 * @param \WP_User $user user.
	 */
	public function can_email_user( $can_email, $campaign_options, $user ) {

		// Abort if already can't email.
		if ( ! $can_email || ( empty( $campaign_options['subscribed'] ) && empty( $campaign_options['not_subscribed'] ) ) ) {
			return $can_email;
		}

		if ( empty( $user->ID ) ) {
			return false;
		}

		return $this->can_email_recipient( $user->ID, $campaign_options );
	}

	/**
	 * Determines whether a subscriber can be emailed for a campaign.
	 *
	 * @param bool               $can_email whether a subscriber can be emailed.
	 * @param array              $campaign_options campaign options.
	 * @param \Hizzle\Noptin\DB\Subscriber $subscriber The subscriber to check.
	 */
	public function can_email_subscriber( $can_email, $campaign_options, $subscriber ) {

		// Abort if already can't email.
		if ( ! $can_email || ( empty( $campaign_options['subscribed'] ) && empty( $campaign_options['not_subscribed'] ) ) ) {
			return $can_email;
		}

		// Abort if we don't have an email.
		$user = get_user_by( 'email', $subscriber->get( 'email' ) );

		if ( empty( $user ) ) {
			return false;
		}

		return $this->can_email_recipient( $user->ID, $campaign_options );
	}

	/**
	 * Determines whether we can email the recipient.
	 *
	 * @param int   $user_id whether a user can be emailed.
	 * @param array $campaign_options campaign options.
	 * @return bool
	 */
	public function can_email_recipient( $user_id, $campaign_options ) {

		if ( ! empty( $campaign_options['subscribed'] ) && ! $this->is_subscribed_to( $user_id, $campaign_options['subscribed'] ) ) {
			return false;
		}

		if ( ! empty( $campaign_options['not_subscribed'] ) && $this->is_subscribed_to( $user_id, $campaign_options['not_subscribed'] ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Determines whether a customer is subscribed to a product.
	 *
	 * @param int    $customer_id customer id.
	 * @param string $product_ids product ids.
	 */
	public function is_subscribed_to( $customer_id, $product_ids ) {

		foreach ( wp_parse_id_list( $product_ids ) as $product_id ) {

			if ( wcs_user_has_subscription( $customer_id, $product_id ) ) {
				return true;
			}
		}

		return false;
	}
}
