<?php

namespace Noptin\WooCommerce\Triggers;

/**
 * Subscription before renewal trigger.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

/**
 * Fires before a subscription renews.
 *
 * @since 1.0.0
 */
class Subscription_Before_Renewal extends Subscription_Trigger {

	/**
	 * Constructor.
	 *
	 * @since 1.0.0
	 * @return string
	 */
	public function __construct() {
		add_action( 'noptin_woocommerce_daily_action', array( $this, 'maybe_trigger' ) );
		add_action( 'handle_noptin_' . $this->get_id() . '_trigger', array( $this, 'trigger_for_rule' ), 10, 2 );
	}

	/**
	 * @inheritdoc
	 */
	public function get_id() {
		return 'woocommerce_subscription_before_renewal';
	}

	/**
	 * @inheritdoc
	 */
	public function get_name() {
		return __( 'WooCommerce Subscription > Before Renewal', 'noptin-woocommerce' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_description() {
		return __( 'X days before a subscription renews', 'noptin-woocommerce' );
	}

	/**
	 * @inheritdoc
	 */
	public function get_rule_description( $rule ) {
		$settings = $rule->trigger_settings;

		/* translators: %s Renewal days */
		return $this->get_days( $settings, __( 'Days before renewal: %s', 'noptin-woocommerce' ) );
	}

	/**
	 * Returns the days before renewal.
	 *
	 * @param array $settings The trigger settings.
	 * @param string $context The context.
	 * @return string
	 */
	public function get_days( $settings, $context ) {

		// Abort if days are not specified.
		if ( empty( $settings['days'] ) ) {
			return $this->get_description();
		}

		// Prepare days array, e.g, 1 day, 3 days, 10 days.
		$_days = array();

		foreach ( array_unique( wp_parse_id_list( $settings['days'] ) ) as $days ) {
			$_days[] = sprintf(
				/* translators: %d number of days */
				_n( '%d day', '%d days', $days, 'noptin-woocommerce' ),
				$days
			);
		}

		// Combine the days.
		$last_day = array_pop( $_days );

		if ( $_days ) {
			$days = sprintf(
				// translators: %s number of days, e.g, 1 day, 3 days, 10 days.
				__( '%1$s and %2$s', 'noptin-woocommerce' ),
				implode( ', ', $_days ),
				$last_day
			);
		} else {
			$days = $last_day;
		}

		// Generate the description.
		return sprintf(
			'%s <p class="description">%s</p>',
			$this->get_description(),
			sprintf( $context, '<code>' . $days . '</code>' )
		);
	}

	/**
	 * @inheritdoc
	 */
	public function get_settings() {

		return array_merge(
			array(
				'days' => array(
					'el'          => 'input',
					'label'       => __( 'Days before renewal', 'noptin-woocommerce' ),
					'description' => __( 'Enter a comma-separated list of days before renewal for which this trigger will fire.', 'noptin-woocommerce' ),
					'placeholder' => '1, 3, 7',
				),
			),
			parent::get_settings()
		);

	}

	/**
	 * Fetch subscriptions for the given number of days.
	 *
	 * @param int $days The days before expiry.
	 * @return int[] $subscription_ids The subscription ids.
	 */
	public function get_subscriptions_for_days( $days ) {

		return get_posts(
			array(
				'post_type'      => 'shop_subscription',
				'post_status'    => 'wc-active',
				'posts_per_page' => -1,
				'fields'         => 'ids',
				'meta_query'     => array(
					array(
						'key'     => wcs_get_date_meta_key( 'next_payment' ),
						'value'   => gmdate( 'Y-m-d 00:00:00', strtotime( "+{$days} days" ) ),
						'compare' => '>=',
						'type'    => 'DATETIME',
					),
					array(
						'key'     => wcs_get_date_meta_key( 'next_payment' ),
						'value'   => gmdate( 'Y-m-d 23:59:59', strtotime( "+{$days} days" ) ),
						'compare' => '<=',
						'type'    => 'DATETIME',
					),
				),
			)
		);
	}

	/**
	 * Fetch subscriptions that are about to renew.
	 *
	 * @param int[] $days The days before expiry.
	 * @return int[] $subscription_ids The subscription ids.
	 */
	public function get_subscriptions( $days ) {
		$subscriptions = array();

		foreach ( $days as $day ) {
			$subscriptions = array_merge( $subscriptions, $this->get_subscriptions_for_days( $day ) );
		}

		return array_unique( wp_parse_id_list( $subscriptions ) );

	}

	/**
	 * Maybe triggers an action.
	 */
	public function maybe_trigger() {

		// Loop through all rules.
		foreach ( $this->get_rules() as $rule ) {

            // Prepare the rule.
            $rule = noptin()->automation_rules->prepare_rule( $rule );

			if ( ! isset( $rule->trigger_settings['days'] ) ) {
				continue;
			}

			// Fetch subscriptions.
			$subscriptions = $this->get_subscriptions( wp_parse_id_list( $rule->trigger_settings['days'] ) );

			// Abort if no subscriptions are found.
			if ( empty( $subscriptions ) ) {
				continue;
			}

			// Trigger for each subscription.
			foreach ( $subscriptions as $index => $subscription_id ) {

				if ( $index > 10 ) {
					$schedule_for = floor( $index / 10 ) * MINUTE_IN_SECONDS;
					schedule_noptin_background_action( time() + $schedule_for, 'handle_noptin_' . $this->get_id() . '_trigger', $subscription_id, $rule->id );
				} else {
					schedule_noptin_background_action( time() + 30, 'handle_noptin_' . $this->get_id() . '_trigger', $subscription_id, $rule->id );
				}
			}
        }

	}

	/**
	 * Triggers a subscription event event.
	 *
	 * @param int $subscription_id The subscription id.
	 * @param int $rule_id The rule id.
	 */
	public function trigger_for_rule( $subscription_id, $rule_id ) {

		$subscription = wcs_get_subscription( $subscription_id );

		if ( ! $subscription || ! $subscription->get_id() ) {
			return;
		}

		$customer = new \WC_Customer( $subscription->get_customer_id() );

		// Set customer data from order if customer is not found.
		if ( ! $customer->get_id() ) {
			$customer->set_email( $subscription->get_billing_email() );
			$customer->set_billing_email( $subscription->get_billing_email() );
			$customer->set_first_name( $subscription->get_billing_first_name() );
			$customer->set_last_name( $subscription->get_billing_last_name() );
		}

		// Prepare the trigger.
		$args = $this->before_trigger_wc( $subscription->get_last_order( 'all' ), $customer );

		// Prepare the trigger.
		self::$subscription = $subscription;

		$args['email']           = $subscription->get_billing_email();
		$args['subscription_id'] = $subscription->get_id();
		$args['rule_id']         = $rule_id;

		$this->trigger( $customer, $args );

		$this->after_trigger_wc( $args );

		self::$subscription = false;
	}

	/**
	 * Triggers action callbacks.
	 *
	 * @since 1.2.8
	 * @param \WC_Customer $subject The subject.
	 * @param array $args Extra arguments passed to the action.
	 * @return void
	 */
	public function trigger( $subject, $args ) {

		if ( empty( $args['rule_id'] ) ) {
			return;
		}

		$args = $this->prepare_trigger_args( $subject, $args );
		$rule = new \Noptin_Automation_Rule( $args['rule_id'] );

		// Retrieve the action.
		$action = noptin()->automation_rules->get_action( $rule->action_id );
		if ( empty( $action ) ) {
			return;
		}

		// Set the current email.
		$GLOBALS['current_noptin_email'] = $this->get_subject_email( $subject, $rule, $args );

		// Are we delaying the action?
		$delay = $rule->get_delay();

		if ( $delay > 0 ) {
			do_action( 'noptin_delay_automation_rule_execution', $rule, $args, $delay );
			return;
		}

		// Ensure that the rule is valid for the provided args.
		if ( $this->is_rule_valid_for_args( $rule, $args, $subject, $action ) ) {
			$action->maybe_run( $subject, $rule, $args );
		}

	}

	/**
	 * Serializes the trigger args.
	 *
	 * @since 1.11.1
	 * @param array $args The args.
	 * @return false|array
	 */
	public function serialize_trigger_args( $args ) {
		return array(
			'subscription_id' => $args['subscription_id'],
			'rule_id'         => $args['rule_id'],
		);
	}
}
