<?php

namespace Noptin\Addons_Pack\Users;

/**
 * Adds WordPress user related improvements.
 *
 * @since 1.0.0
 */

defined( 'ABSPATH' ) || exit;

define( 'NOPTIN_WP_USERS_VERSION', '2.0.0' );
define( 'NOPTIN_WP_USERS_FILE', __FILE__ );

/**
 * Adds WordPress user related improvements.
 *
 */
class Main {

	/**
	 * Stores the main component instance.
	 *
	 * @access private
	 * @var    Main $instance The main component instance.
	 * @since  1.0.0
	 */
	private static $instance = null;

	/**
	 * Paid Memberships Pro integration.
	 *
	 * @var    PMPro $pmpro Paid Memberships Pro integration.
	 * @since  1.0.0
	 */
	public $pmpro;

	/**
	 * Get active instance
	 *
	 * @access public
	 * @since  1.0.0
	 * @return Main The main component instance.
	 */
	public static function instance() {

		if ( empty( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		add_filter( 'noptin_automation_rules_load', array( $this, 'load_triggers' ) );
		add_filter( 'noptin_automation_rules_load', array( $this, 'load_actions' ) );
		add_filter( 'noptin_automation_trigger_known_smart_tags', '\Noptin\Addons_Pack\Users\Custom_Fields::filter_conditional_logic_rules', 10, 2 );
		add_action( 'plugins_loaded', array( $this, 'load_pmpro' ) );
		add_action( 'noptin_email_manager_init', array( $this, 'load_mailer' ) );

		// Allow sending emails to users.
		add_filter( 'bulk_actions-users', array( $this, 'add_bulk_email_action' ) );
		add_filter( 'handle_bulk_actions-users', array( $this, 'handle_bulk_email_action' ), 10, 3 );
		add_filter( 'user_row_actions', array( $this, 'add_row_email_action' ), 10, 2 );
	}

	/**
	 * Loads automation rule triggers.
	 *
	 * @param \Noptin_Automation_Rules $rules
	 */
	public function load_triggers( $rules ) {

		$rules->add_trigger( new Triggers\New_User() );
		$rules->add_trigger( new Triggers\Update_User() );
		$rules->add_trigger( new Triggers\Delete_User() );
		$rules->add_trigger( new Triggers\Add_User_Role() );
		$rules->add_trigger( new Triggers\Remove_User_Role() );
		$rules->add_trigger( new Triggers\Login() );
	}

	/**
	 * Loads automation rule actions.
	 *
	 * @param \Noptin_Automation_Rules $rules
	 */
	public function load_actions( $rules ) {

		$rules->add_action( new Actions\Add_User_Role() );
		$rules->add_action( new Actions\Remove_User_Role() );
		$rules->add_action( new Actions\Set_User_Role() );
		$rules->add_action( new Actions\Delete_User() );
		$rules->add_action( new Actions\Add_User() );
	}

	/**
	 * Loads Paid Memberships Pro integration.
	 */
	public function load_pmpro() {

		if ( defined( 'PMPRO_VERSION' ) ) {
			$this->pmpro = new PMPro();
		}
	}

	/**
	 * Loads the mailer.
	 *
	 * @param \Noptin_Email_Manager $manager
	 */
	public function load_mailer( $manager ) {
		$manager->mass_mailer_users = new Mailer();
	}

	/**
	 * Adds a bulk email action to the users screen.
	 *
	 * @param array $actions
	 * @return array
	 */
	public function add_bulk_email_action( $actions ) {

		$actions['noptin_send_email'] = __( 'Send Email', 'noptin-addons-pack' );
		return $actions;
	}

	/**
	 * Handles the bulk email action.
	 *
	 * @param string $redirect_to
	 * @param string $action
	 * @param array $user_ids
	 * @return string
	 */
	public function handle_bulk_email_action( $redirect_to, $action, $user_ids ) {

		if ( 'noptin_send_email' !== $action ) {
			return $redirect_to;
		}

		return get_noptin_email_recipients_url( $user_ids, 'wp_users' );
	}

	/**
	 * Adds a row email action to the users screen.
	 *
	 * @param array $actions
	 * @param object $user
	 * @return array
	 */
	public function add_row_email_action( $actions, $user ) {

		$actions['noptin_send_email'] = sprintf(
			'<a href="%s" style="color: #1b5e20;font-weight: 600;">%s</a>',
			esc_url( get_noptin_email_recipients_url( $user->ID, 'wp_users' ) ),
			__( 'Send Email', 'noptin-addons-pack' )
		);

		return $actions;
	}
}
